/*

   Derby - Class org.apache.derby.shared.common.stream.HeaderPrintWriter

   Licensed to the Apache Software Foundation (ASF) under one or more
   contributor license agreements.  See the NOTICE file distributed with
   this work for additional information regarding copyright ownership.
   The ASF licenses this file to you under the Apache License, Version 2.0
   (the "License"); you may not use this file except in compliance with
   the License.  You may obtain a copy of the License at

      http://www.apache.org/licenses/LICENSE-2.0

   Unless required by applicable law or agreed to in writing, software
   distributed under the License is distributed on an "AS IS" BASIS,
   WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
   See the License for the specific language governing permissions and
   limitations under the License.

 */

package org.apache.derby.shared.common.stream;

import java.io.PrintWriter;

/**
 * A HeaderPrintWriter is like a PrintWriter with support for
 * including a header in the output. It is expected users
 * will use HeaderPrintWriters to prepend headings to trace
 * and log messages.
 * 
 */
public interface HeaderPrintWriter
{
	/**
	 * Puts out some setup info for
	 * the current write and the write(s) that will be put out next.
	 * It ends with a \n\r.
	 * <p>
	 * All other writes to the stream use the
	 * PrintStream interface.
     *
     * @param message Message to print
	 */
	public void printlnWithHeader(String message);

	/**
	 * Return the header for the stream.
     *
     * @return the stream header
	 */
	public PrintWriterGetHeader getHeader();
	
	/**
	 * Gets a PrintWriter object for writing to this HeaderPrintWriter.
	 * Users may use the HeaderPrintWriter to access methods not included
	 * in this interface or to invoke methods or constructors which require
	 * a PrintWriter. 
	 *
	 * Interleaving calls to a printWriter and its associated HeaderPrintWriter
	 * is not supported.
	 *
     * @return the print machinery
	 */
	public PrintWriter getPrintWriter();

	/**
	 * Gets the name of the wrapped writer or stream
     *
     * @return the name of the wrapped writer or stream
	 */
	public String getName ();

	/*
	 * The routines that mimic java.io.PrintWriter...
	 */
	/**
     * Print a message.
     *
     * @param message The message to print
     *
	 * @see java.io.PrintWriter#print
	 */
	public void print(String message);

	/**
     * Print a message with a newline.
     *
     * @param message The message to print
     *
	 * @see java.io.PrintWriter#println
	 */
	public void println(String message);

	/**
     * Print a message with a newline.
     *
     * @param message The message to print
     *
	 * @see java.io.PrintWriter#println
	 */
	public void println(Object message);

	/**
	* @see java.io.PrintWriter#flush
	 */
	public void flush();
}

