/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.rya.indexing.pcj.fluo.app;

import static com.google.common.base.Preconditions.checkNotNull;
import static org.apache.rya.indexing.pcj.fluo.app.IncrementalUpdateConstants.NODEID_BS_DELIM;

import java.util.Objects;

import org.apache.fluo.api.data.Bytes;
import org.apache.rya.indexing.pcj.fluo.app.util.BindingHashShardingFunction;

import edu.umd.cs.findbugs.annotations.DefaultAnnotation;
import edu.umd.cs.findbugs.annotations.NonNull;
import net.jcip.annotations.Immutable;

/**
 * The values of an Accumulo Row ID for a row that stores a Binding set for a specific Node ID of a query.
 */
@Immutable
@DefaultAnnotation(NonNull.class)
public class BindingSetRow {
    private final String nodeId;
    private final String bindingSetString;

    /**
     * Constructs an instance of {@link BindingSetRow}.
     *
     * @param nodeId - The Node ID of a query node. (not null)
     * @param bindingSetString - A Binding Set that is part of the node's results. (not null)
     */
    public BindingSetRow(final String nodeId, final String bindingSetString) {
        this.nodeId = checkNotNull(nodeId);
        this.bindingSetString = checkNotNull(bindingSetString);
    }

    /**
     * @return The Node ID of a query node.
     */
    public String getNodeId() {
        return nodeId;
    }

    /**
     * @return A Binding Set that is part of the node's results.
     */
    public String getBindingSetString() {
        return bindingSetString;
    }

    /**
     * Parses the {@link Bytes} of an Accumulo Row ID into a {@link BindingSetRow}.
     *
     * @param row - The Row ID to parse. (not null).
     * @return A {@link BindingSetRow} holding the parsed values.
     */
    public static BindingSetRow make(final Bytes row) {
        checkNotNull(row);

        // Read the Node ID from the row's bytes.
        final String[] rowArray = row.toString().split(NODEID_BS_DELIM);
        final String nodeId = rowArray[0];
        final String bindingSetString = rowArray.length == 2 ? rowArray[1] : "";
        return new BindingSetRow(nodeId, bindingSetString);
    }

    /**
     * Creates a BindingSetRow from a sharded row entry, where the row is sharded according to
     * {@link BindingHashShardingFunction}.
     *
     * @param prefixBytes - prefix of the node type that the row corresponds to (see prefixes in
     *            {@link IncrementalUpdateConstants}).
     * @param row - row that BindingSetRow is created from
     * @return - BindingSetRow object
     */
    public static BindingSetRow makeFromShardedRow(Bytes prefixBytes, Bytes row) {
        return make(BindingHashShardingFunction.removeHash(prefixBytes, row));
    }

    @Override
    public String toString() {
        return "NodeId: " + nodeId + "\n" + "BindingSet String: " + bindingSetString;
    }

    @Override
    public boolean equals(Object other) {
        if(this == other) { return true;}

        if (other instanceof BindingSetRow) {
            BindingSetRow row = (BindingSetRow) other;
            return Objects.equals(nodeId, row.nodeId) && Objects.equals(bindingSetString, row.bindingSetString);
        }

        return false;
    }

    @Override
    public int hashCode() {
        return Objects.hash(nodeId, bindingSetString);
    }
}