﻿#region License

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

#endregion

using System;
using System.Collections.Generic;

namespace Gremlin.Net.Structure
{
    /// <summary>
    ///     A common base class for Graph elements.
    /// </summary>
    public abstract class Element : IEquatable<Element>
    {
        /// <summary>
        ///     Initializes a new instance of the <see cref="Element" /> class.
        /// </summary>
        /// <param name="id">The id of the element.</param>
        /// <param name="label">The label of the element.</param>
        /// <param name="properties">Optional properties of the element.</param>
        protected Element(object? id, string label, dynamic[]? properties = null)
        {
            Id = id;
            Label = label;
            Properties = properties ?? Array.Empty<dynamic>();
        }

        /// <summary>
        ///     Gets the id of this <see cref="Element" />.
        /// </summary>
        public object? Id { get; }

        /// <summary>
        ///     Gets the label of this <see cref="Element" />.
        /// </summary>
        public string Label { get; }

        /// <summary>
        ///     Gets the properties of this <see cref="Element" />.
        /// </summary>
        public dynamic[] Properties { get; }

        /// <inheritdoc />
        public bool Equals(Element? other)
        {
            if (ReferenceEquals(null, other)) return false;
            if (ReferenceEquals(this, other)) return true;
            return Equals(Id, other.Id);
        }

        /// <inheritdoc />
        public override bool Equals(object? obj)
        {
            if (ReferenceEquals(null, obj)) return false;
            if (ReferenceEquals(this, obj)) return true;
            if (obj.GetType() != GetType()) return false;
            return Equals((Element) obj);
        }

        /// <inheritdoc />
        public override int GetHashCode()
        {
            return Id != null ? Id.GetHashCode() : 0;
        }
    }
}