/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.common.blobstore.fs;

import java.io.BufferedInputStream;
import java.io.FileNotFoundException;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.file.DirectoryStream;
import java.nio.file.FileAlreadyExistsException;
import java.nio.file.FileVisitResult;
import java.nio.file.FileVisitor;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.NoSuchFileException;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.SimpleFileVisitor;
import java.nio.file.StandardCopyOption;
import java.nio.file.StandardOpenOption;
import java.nio.file.attribute.BasicFileAttributes;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.concurrent.atomic.AtomicLong;
import org.elasticsearch.common.UUIDs;
import org.elasticsearch.common.blobstore.BlobContainer;
import org.elasticsearch.common.blobstore.BlobMetaData;
import org.elasticsearch.common.blobstore.BlobPath;
import org.elasticsearch.common.blobstore.DeleteResult;
import org.elasticsearch.common.blobstore.fs.FsBlobStore;
import org.elasticsearch.common.blobstore.support.AbstractBlobContainer;
import org.elasticsearch.common.blobstore.support.PlainBlobMetaData;
import org.elasticsearch.core.internal.io.IOUtils;
import org.elasticsearch.core.internal.io.Streams;

public class FsBlobContainer
extends AbstractBlobContainer {
    private static final String TEMP_FILE_PREFIX = "pending-";
    protected final FsBlobStore blobStore;
    protected final Path path;

    public FsBlobContainer(FsBlobStore blobStore, BlobPath blobPath, Path path) {
        super(blobPath);
        this.blobStore = blobStore;
        this.path = path;
    }

    @Override
    public Map<String, BlobMetaData> listBlobs() throws IOException {
        return this.listBlobsByPrefix(null);
    }

    @Override
    public Map<String, BlobContainer> children() throws IOException {
        HashMap<String, FsBlobContainer> builder = new HashMap<String, FsBlobContainer>();
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.path);){
            for (Path file : stream) {
                if (!Files.isDirectory(file, new LinkOption[0])) continue;
                String name = file.getFileName().toString();
                builder.put(name, new FsBlobContainer(this.blobStore, this.path().add(name), file));
            }
        }
        return Collections.unmodifiableMap(builder);
    }

    @Override
    public Map<String, BlobMetaData> listBlobsByPrefix(String blobNamePrefix) throws IOException {
        HashMap<String, PlainBlobMetaData> builder = new HashMap<String, PlainBlobMetaData>();
        blobNamePrefix = blobNamePrefix == null ? "" : blobNamePrefix;
        try (DirectoryStream<Path> stream = Files.newDirectoryStream(this.path, blobNamePrefix + "*");){
            for (Path file : stream) {
                BasicFileAttributes attrs = Files.readAttributes(file, BasicFileAttributes.class, new LinkOption[0]);
                if (!attrs.isRegularFile()) continue;
                builder.put(file.getFileName().toString(), new PlainBlobMetaData(file.getFileName().toString(), attrs.size()));
            }
        }
        return Collections.unmodifiableMap(builder);
    }

    @Override
    public void deleteBlob(String blobName) throws IOException {
        Path blobPath = this.path.resolve(blobName);
        if (Files.isDirectory(blobPath, new LinkOption[0])) {
            Files.walkFileTree(blobPath, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

                @Override
                public FileVisitResult postVisitDirectory(Path dir, IOException exc) throws IOException {
                    Files.delete(dir);
                    return FileVisitResult.CONTINUE;
                }
            });
        } else {
            Files.delete(blobPath);
        }
    }

    @Override
    public DeleteResult delete() throws IOException {
        final AtomicLong filesDeleted = new AtomicLong(0L);
        final AtomicLong bytesDeleted = new AtomicLong(0L);
        Files.walkFileTree(this.path, (FileVisitor<? super Path>)new SimpleFileVisitor<Path>(){

            @Override
            public FileVisitResult postVisitDirectory(Path dir, IOException impossible) throws IOException {
                assert (impossible == null);
                Files.delete(dir);
                return FileVisitResult.CONTINUE;
            }

            @Override
            public FileVisitResult visitFile(Path file, BasicFileAttributes attrs) throws IOException {
                Files.delete(file);
                filesDeleted.incrementAndGet();
                bytesDeleted.addAndGet(attrs.size());
                return FileVisitResult.CONTINUE;
            }
        });
        return new DeleteResult(filesDeleted.get(), bytesDeleted.get());
    }

    @Override
    public InputStream readBlob(String name) throws IOException {
        Path resolvedPath = this.path.resolve(name);
        try {
            return new BufferedInputStream(Files.newInputStream(resolvedPath, new OpenOption[0]), this.blobStore.bufferSizeInBytes());
        }
        catch (FileNotFoundException fnfe) {
            throw new NoSuchFileException("[" + name + "] blob not found");
        }
    }

    @Override
    public void writeBlob(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        if (!failIfAlreadyExists) {
            this.deleteBlobIgnoringIfNotExists(blobName);
        }
        Path file = this.path.resolve(blobName);
        try (OutputStream outputStream = Files.newOutputStream(file, StandardOpenOption.CREATE_NEW);){
            Streams.copy(inputStream, outputStream);
        }
        IOUtils.fsync(file, false);
        IOUtils.fsync(this.path, true);
    }

    @Override
    public void writeBlobAtomic(String blobName, InputStream inputStream, long blobSize, boolean failIfAlreadyExists) throws IOException {
        String tempBlob = FsBlobContainer.tempBlobName(blobName);
        Path tempBlobPath = this.path.resolve(tempBlob);
        try {
            try (OutputStream outputStream = Files.newOutputStream(tempBlobPath, StandardOpenOption.CREATE_NEW);){
                Streams.copy(inputStream, outputStream);
            }
            IOUtils.fsync(tempBlobPath, false);
            this.moveBlobAtomic(tempBlob, blobName, failIfAlreadyExists);
        }
        catch (IOException ex) {
            try {
                this.deleteBlobIgnoringIfNotExists(tempBlob);
            }
            catch (IOException e) {
                ex.addSuppressed(e);
            }
            throw ex;
        }
        finally {
            IOUtils.fsync(this.path, true);
        }
    }

    public void moveBlobAtomic(String sourceBlobName, String targetBlobName, boolean failIfAlreadyExists) throws IOException {
        Path sourceBlobPath = this.path.resolve(sourceBlobName);
        Path targetBlobPath = this.path.resolve(targetBlobName);
        if (Files.exists(targetBlobPath, new LinkOption[0])) {
            if (failIfAlreadyExists) {
                throw new FileAlreadyExistsException("blob [" + targetBlobPath + "] already exists, cannot overwrite");
            }
            this.deleteBlobIgnoringIfNotExists(targetBlobName);
        }
        Files.move(sourceBlobPath, targetBlobPath, StandardCopyOption.ATOMIC_MOVE);
    }

    public static String tempBlobName(String blobName) {
        return TEMP_FILE_PREFIX + blobName + "-" + UUIDs.randomBase64UUID();
    }

    public static boolean isTempBlobName(String blobName) {
        return blobName.startsWith(TEMP_FILE_PREFIX);
    }
}

