/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements. See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership. The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License. You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied. See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
package org.apache.axiom.buildutils.enforcer;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.util.Set;

import org.apache.maven.enforcer.rule.api.EnforcerRule;
import org.apache.maven.enforcer.rule.api.EnforcerRuleException;
import org.apache.maven.enforcer.rule.api.EnforcerRuleHelper;
import org.codehaus.plexus.component.configurator.expression.ExpressionEvaluationException;
import org.codehaus.plexus.util.DirectoryScanner;
import org.objectweb.asm.ClassReader;

public class NoPackageCyclesEnforcerRule implements EnforcerRule {

    @Override
    public void execute(EnforcerRuleHelper helper) throws EnforcerRuleException {
        try {
            File classesDir = new File((String)helper.evaluate("${project.build.outputDirectory}"));
            if (!classesDir.exists()) {
                return;
            }
            DirectoryScanner ds = new DirectoryScanner();
            ds.setIncludes(new String[] { "**/*.class" });
            ds.setBasedir(classesDir);
            ds.scan();
            ReferenceCollector referenceCollector = new ReferenceCollector();
            for (String relativePath : ds.getIncludedFiles()) {
                try {
                    InputStream in = new FileInputStream(new File(classesDir, relativePath));
                    try {
                        new ClassReader(in).accept(new ClassProcessor(referenceCollector), ClassReader.SKIP_DEBUG | ClassReader.SKIP_FRAMES);
                    } finally {
                        in.close();
                    }
                } catch (IOException ex) {
                    throw new EnforcerRuleException("Failed to read " + relativePath + ": " + ex.getMessage(), ex);
                }
            }
            Set<Reference> references = referenceCollector.getClassReferencesForPackageCycle();
            if (references != null) {
                StringBuilder buffer = new StringBuilder("Package cycle detected. Classes involved:");
                for (Reference reference : references) {
                    buffer.append("\n  ");
                    buffer.append(reference.getFrom());
                    buffer.append(" -> ");
                    buffer.append(reference.getTo());
                }
                throw new EnforcerRuleException(buffer.toString());
            }
        } catch (ExpressionEvaluationException ex) {
            throw new EnforcerRuleException("Failed to evaluate expression: " + ex.getMessage(), ex);
        }
    }

    @Override
    public boolean isCacheable() {
        return false;
    }

    @Override
    public boolean isResultValid(EnforcerRule cachedRule) {
        return false;
    }

    @Override
    public String getCacheId() {
        return null;
    }
}
