/* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
 * Copyright by The HDF Group.                                               *
 * All rights reserved.                                                      *
 *                                                                           *
 * This file is part of HDF5.  The full HDF5 copyright notice, including     *
 * terms governing use, modification, and redistribution, is contained in    *
 * the LICENSE file, which can be found at the root of the source code       *
 * distribution tree, or in https://www.hdfgroup.org/licenses.               *
 * If you do not have access to either file, you may request a copy from     *
 * help@hdfgroup.org.                                                        *
 * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */

#ifndef H5CXprivate_H
#define H5CXprivate_H

#include "H5private.h"   
#include "H5ACprivate.h" 
#ifdef H5_HAVE_PARALLEL
#include "H5FDprivate.h" 
#endif                   
#include "H5Pprivate.h"  
#include "H5Tprivate.h"  
#include "H5Tconv.h"     
#include "H5Zprivate.h"  

typedef struct H5CX_state_t {
    hid_t                 dcpl_id;            
    hid_t                 dxpl_id;            
    hid_t                 lapl_id;            
    hid_t                 lcpl_id;            
    void                 *vol_wrap_ctx;       
    H5VL_connector_prop_t vol_connector_prop; 

#ifdef H5_HAVE_PARALLEL
    
    bool coll_metadata_read; 
#endif                       
} H5CX_state_t;

typedef struct H5CX_t {
    
    hid_t           dxpl_id; 
    H5P_genplist_t *dxpl;    

    
    hid_t           lcpl_id; 
    H5P_genplist_t *lcpl;    

    
    hid_t           lapl_id; 
    H5P_genplist_t *lapl;    

    
    hid_t           dcpl_id; 
    H5P_genplist_t *dcpl;    

    
    hid_t           dapl_id; 
    H5P_genplist_t *dapl;    

    
    hid_t           fapl_id; 
    H5P_genplist_t *fapl;    

    
    haddr_t tag; 

    
    H5AC_ring_t ring; 

#ifdef H5_HAVE_PARALLEL
    
    bool         coll_metadata_read; 
    MPI_Datatype btype;              
    MPI_Datatype ftype;              
    bool         mpi_file_flushing;  
    bool         rank0_bcast;        
#endif                               

    
    size_t    max_temp_buf;            
    bool      max_temp_buf_valid;      
    void     *tconv_buf;               
    bool      tconv_buf_valid;         
    void     *bkgr_buf;                
    bool      bkgr_buf_valid;          
    H5T_bkg_t bkgr_buf_type;           
    bool      bkgr_buf_type_valid;     
    double    btree_split_ratio[3];    
    bool      btree_split_ratio_valid; 
    size_t    vec_size;                
    bool      vec_size_valid;          
#ifdef H5_HAVE_PARALLEL
    H5FD_mpio_xfer_t io_xfer_mode; 
    bool             io_xfer_mode_valid;      
    H5FD_mpio_collective_opt_t mpio_coll_opt; 
    bool mpio_coll_opt_valid;                 
    H5FD_mpio_chunk_opt_t
             mpio_chunk_opt_mode;        
    bool     mpio_chunk_opt_mode_valid;  
    unsigned mpio_chunk_opt_num;         
    bool     mpio_chunk_opt_num_valid;   
    unsigned mpio_chunk_opt_ratio;       
    bool     mpio_chunk_opt_ratio_valid; 
#endif                                   
    H5Z_EDC_t               err_detect;  
    bool                    err_detect_valid;     
    H5Z_cb_t                filter_cb;            
    bool                    filter_cb_valid;      
    H5Z_data_xform_t       *data_transform;       
    bool                    data_transform_valid; 
    H5T_vlen_alloc_info_t   vl_alloc_info;        
    bool                    vl_alloc_info_valid;  
    H5T_conv_cb_t           dt_conv_cb;           
    bool                    dt_conv_cb_valid;     
    H5D_selection_io_mode_t selection_io_mode;    
    bool                    selection_io_mode_valid; 
    bool modify_write_buf; 
    bool modify_write_buf_valid; 

    
#ifdef H5_HAVE_PARALLEL
    H5D_mpio_actual_chunk_opt_mode_t mpio_actual_chunk_opt; 
    bool mpio_actual_chunk_opt_set; 
    H5D_mpio_actual_io_mode_t
             mpio_actual_io_mode; 
    bool     mpio_actual_io_mode_set;        
    uint32_t mpio_local_no_coll_cause;       
    bool     mpio_local_no_coll_cause_set;   
    bool     mpio_local_no_coll_cause_valid; 
    uint32_t mpio_global_no_coll_cause;      
    bool mpio_global_no_coll_cause_set;      
    bool mpio_global_no_coll_cause_valid;    
#ifdef H5_HAVE_INSTRUMENTED_LIBRARY
    int mpio_coll_chunk_link_hard;       
    bool mpio_coll_chunk_link_hard_set;  
    int  mpio_coll_chunk_multi_hard;     
    bool mpio_coll_chunk_multi_hard_set; 
    int  mpio_coll_chunk_link_num_true;  
    bool mpio_coll_chunk_link_num_true_set;    
    int mpio_coll_chunk_link_num_false;        
    bool mpio_coll_chunk_link_num_false_set;   
    int mpio_coll_chunk_multi_ratio_coll;      
    bool mpio_coll_chunk_multi_ratio_coll_set; 
    int mpio_coll_chunk_multi_ratio_ind;       
    bool mpio_coll_chunk_multi_ratio_ind_set;  
    bool mpio_coll_rank0_bcast;                
    bool mpio_coll_rank0_bcast_set;   
#endif                                
#endif                                
    uint32_t no_selection_io_cause;   
    bool no_selection_io_cause_set;   
    bool no_selection_io_cause_valid; 

    uint32_t actual_selection_io_mode;   
    bool actual_selection_io_mode_set;   
    bool actual_selection_io_mode_valid; 

    
    H5T_cset_t encoding;         
    bool       encoding_valid;   
    unsigned intermediate_group; 
    bool     intermediate_group_valid; 

    
    size_t nlinks;       
    bool   nlinks_valid; 

    
    bool    do_min_dset_ohdr; 
    bool    do_min_dset_ohdr_valid; 
    uint8_t ohdr_flags;             
    bool    ohdr_flags_valid;       

    
    const char *extfile_prefix;       
    bool        extfile_prefix_valid; 
    const char *vds_prefix;           
    bool        vds_prefix_valid;     

    
    H5F_libver_t low_bound;       
    bool         low_bound_valid; 
    H5F_libver_t high_bound;      
    bool high_bound_valid;        

    
    H5VL_connector_prop_t vol_connector_prop; 
    bool  vol_connector_prop_valid;           
    void *vol_wrap_ctx;                       
    bool  vol_wrap_ctx_valid; 
} H5CX_t;

typedef struct H5CX_node_t {
    H5CX_t              ctx;  
    struct H5CX_node_t *next; 
} H5CX_node_t;

H5_DLL herr_t H5CX_push(H5CX_node_t *cnode);
H5_DLL herr_t H5CX_pop(bool update_dxpl_props);
H5_DLL bool   H5CX_pushed(void);
H5_DLL bool   H5CX_is_def_dxpl(void);

H5_DLL herr_t H5CX_retrieve_state(H5CX_state_t **api_state);
H5_DLL herr_t H5CX_restore_state(const H5CX_state_t *api_state);
H5_DLL herr_t H5CX_free_state(H5CX_state_t *api_state);

H5_DLL void   H5CX_set_dxpl(hid_t dxpl_id);
H5_DLL void   H5CX_set_lcpl(hid_t lcpl_id);
H5_DLL void   H5CX_set_lapl(hid_t lapl_id);
H5_DLL void   H5CX_set_dcpl(hid_t dcpl_id);
H5_DLL herr_t H5CX_set_libver_bounds(H5F_t *f);
H5_DLL herr_t H5CX_set_apl(hid_t *acspl_id, const H5P_libclass_t *libclass, hid_t loc_id, bool is_collective);
H5_DLL herr_t H5CX_set_loc(hid_t loc_id);
H5_DLL herr_t H5CX_set_vol_wrap_ctx(void *wrap_ctx);
H5_DLL herr_t H5CX_set_vol_connector_prop(const H5VL_connector_prop_t *vol_connector_prop);

H5_DLL hid_t       H5CX_get_dxpl(void);
H5_DLL hid_t       H5CX_get_lapl(void);
H5_DLL herr_t      H5CX_get_vol_wrap_ctx(void **wrap_ctx);
H5_DLL herr_t      H5CX_get_vol_connector_prop(H5VL_connector_prop_t *vol_connector_prop);
H5_DLL haddr_t     H5CX_get_tag(void);
H5_DLL H5AC_ring_t H5CX_get_ring(void);
#ifdef H5_HAVE_PARALLEL
H5_DLL bool   H5CX_get_coll_metadata_read(void);
H5_DLL herr_t H5CX_get_mpi_coll_datatypes(MPI_Datatype *btype, MPI_Datatype *ftype);
H5_DLL bool   H5CX_get_mpi_file_flushing(void);
H5_DLL bool   H5CX_get_mpio_rank0_bcast(void);
#endif 

H5_DLL herr_t H5CX_get_btree_split_ratios(double split_ratio[3]);
H5_DLL herr_t H5CX_get_max_temp_buf(size_t *max_temp_buf);
H5_DLL herr_t H5CX_get_tconv_buf(void **tconv_buf);
H5_DLL herr_t H5CX_get_bkgr_buf(void **bkgr_buf);
H5_DLL herr_t H5CX_get_bkgr_buf_type(H5T_bkg_t *bkgr_buf_type);
H5_DLL herr_t H5CX_get_vec_size(size_t *vec_size);
#ifdef H5_HAVE_PARALLEL
H5_DLL herr_t H5CX_get_io_xfer_mode(H5FD_mpio_xfer_t *io_xfer_mode);
H5_DLL herr_t H5CX_get_mpio_coll_opt(H5FD_mpio_collective_opt_t *mpio_coll_opt);
H5_DLL herr_t H5CX_get_mpio_local_no_coll_cause(uint32_t *mpio_local_no_coll_cause);
H5_DLL herr_t H5CX_get_mpio_global_no_coll_cause(uint32_t *mpio_global_no_coll_cause);
H5_DLL herr_t H5CX_get_mpio_chunk_opt_mode(H5FD_mpio_chunk_opt_t *mpio_chunk_opt_mode);
H5_DLL herr_t H5CX_get_mpio_chunk_opt_num(unsigned *mpio_chunk_opt_num);
H5_DLL herr_t H5CX_get_mpio_chunk_opt_ratio(unsigned *mpio_chunk_opt_ratio);
#endif 
H5_DLL herr_t H5CX_get_err_detect(H5Z_EDC_t *err_detect);
H5_DLL herr_t H5CX_get_filter_cb(H5Z_cb_t *filter_cb);
H5_DLL herr_t H5CX_get_data_transform(H5Z_data_xform_t **data_transform);
H5_DLL herr_t H5CX_get_vlen_alloc_info(H5T_vlen_alloc_info_t *vl_alloc_info);
H5_DLL herr_t H5CX_get_dt_conv_cb(H5T_conv_cb_t *cb_struct);
H5_DLL herr_t H5CX_get_selection_io_mode(H5D_selection_io_mode_t *selection_io_mode);
H5_DLL herr_t H5CX_get_no_selection_io_cause(uint32_t *no_selection_io_cause);
H5_DLL herr_t H5CX_get_actual_selection_io_mode(uint32_t *actual_selection_io_mode);
H5_DLL herr_t H5CX_get_modify_write_buf(bool *modify_write_buf);

H5_DLL herr_t H5CX_get_encoding(H5T_cset_t *encoding);
H5_DLL herr_t H5CX_get_intermediate_group(unsigned *crt_intermed_group);

H5_DLL herr_t H5CX_get_nlinks(size_t *nlinks);

H5_DLL herr_t H5CX_get_dset_min_ohdr_flag(bool *dset_min_ohdr_flag);
H5_DLL herr_t H5CX_get_ohdr_flags(uint8_t *ohdr_flags);

H5_DLL herr_t H5CX_get_ext_file_prefix(const char **prefix_extfile);
H5_DLL herr_t H5CX_get_vds_prefix(const char **prefix_vds);

H5_DLL herr_t H5CX_get_libver_bounds(H5F_libver_t *low_bound, H5F_libver_t *high_bound);

H5_DLL void H5CX_set_tag(haddr_t tag);
H5_DLL void H5CX_set_ring(H5AC_ring_t ring);
#ifdef H5_HAVE_PARALLEL
H5_DLL void   H5CX_set_coll_metadata_read(bool cmdr);
H5_DLL herr_t H5CX_set_mpi_coll_datatypes(MPI_Datatype btype, MPI_Datatype ftype);
H5_DLL herr_t H5CX_set_mpio_coll_opt(H5FD_mpio_collective_opt_t mpio_coll_opt);
H5_DLL void   H5CX_set_mpi_file_flushing(bool flushing);
H5_DLL void   H5CX_set_mpio_rank0_bcast(bool rank0_bcast);
#endif 

#ifdef H5_HAVE_PARALLEL
H5_DLL herr_t H5CX_set_io_xfer_mode(H5FD_mpio_xfer_t io_xfer_mode);
#endif 
H5_DLL herr_t H5CX_set_vlen_alloc_info(H5MM_allocate_t alloc_func, void *alloc_info, H5MM_free_t free_func,
                                       void *free_info);

H5_DLL herr_t H5CX_set_nlinks(size_t nlinks);

H5_DLL void H5CX_set_no_selection_io_cause(uint32_t no_selection_io_cause);
H5_DLL void H5CX_set_actual_selection_io_mode(uint32_t actual_selection_io_mode);

#ifdef H5_HAVE_PARALLEL
H5_DLL void H5CX_set_mpio_actual_chunk_opt(H5D_mpio_actual_chunk_opt_mode_t chunk_opt);
H5_DLL void H5CX_set_mpio_actual_io_mode(H5D_mpio_actual_io_mode_t actual_io_mode);
H5_DLL void H5CX_set_mpio_local_no_coll_cause(uint32_t mpio_local_no_coll_cause);
H5_DLL void H5CX_set_mpio_global_no_coll_cause(uint32_t mpio_global_no_coll_cause);
#ifdef H5_HAVE_INSTRUMENTED_LIBRARY
H5_DLL herr_t H5CX_test_set_mpio_coll_chunk_link_hard(int mpio_coll_chunk_link_hard);
H5_DLL herr_t H5CX_test_set_mpio_coll_chunk_multi_hard(int mpio_coll_chunk_multi_hard);
H5_DLL herr_t H5CX_test_set_mpio_coll_chunk_link_num_true(int mpio_coll_chunk_link_num_true);
H5_DLL herr_t H5CX_test_set_mpio_coll_chunk_link_num_false(int mpio_coll_chunk_link_num_false);
H5_DLL herr_t H5CX_test_set_mpio_coll_chunk_multi_ratio_coll(int mpio_coll_chunk_multi_ratio_coll);
H5_DLL herr_t H5CX_test_set_mpio_coll_chunk_multi_ratio_ind(int mpio_coll_chunk_multi_ratio_ind);
H5_DLL herr_t H5CX_test_set_mpio_coll_rank0_bcast(bool rank0_bcast);
#endif 
#endif 

#endif 
