/* -*- tab-width: 4 -*-
 *
 * Electric(tm) VLSI Design System
 *
 * File: Spread.java
 *
 * Copyright (c) 2004 Sun Microsystems and Static Free Software
 *
 * Electric(tm) is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * Electric(tm) is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with Electric(tm); see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, Mass 02111-1307, USA.
 */
package com.sun.electric.tool.user.dialogs;

import com.sun.electric.database.constraint.Layout;
import com.sun.electric.database.geometry.Geometric;
import com.sun.electric.database.hierarchy.Cell;
import com.sun.electric.database.prototype.NodeProto;
import com.sun.electric.database.text.TextUtils;
import com.sun.electric.database.topology.NodeInst;
import com.sun.electric.database.topology.ArcInst;
import com.sun.electric.database.topology.Connection;
import com.sun.electric.database.variable.FlagSet;
import com.sun.electric.technology.Technology;
import com.sun.electric.technology.SizeOffset;
import com.sun.electric.technology.PrimitiveNode;
import com.sun.electric.tool.Job;
import com.sun.electric.tool.user.User;
import com.sun.electric.tool.user.CircuitChanges;
import com.sun.electric.tool.user.Highlight;
import com.sun.electric.tool.user.ui.TopLevel;
import com.sun.electric.tool.user.ui.WindowFrame;

import java.awt.Font;
import java.awt.GraphicsEnvironment;
import java.awt.event.ActionListener;
import java.awt.event.ActionEvent;
import java.awt.geom.Point2D;
import java.awt.image.DataBufferByte;
import java.awt.image.Raster;
import java.util.Iterator;


/**
 * Class to handle the "Spread" dialog.
 */
public class Spread extends EDialog
{
	public static void showSpreadDialog()
	{
		NodeInst ni = (NodeInst)Highlight.getOneElectricObject(NodeInst.class);
		if (ni == null) return;

		Spread dialog = new Spread(TopLevel.getCurrentJFrame(), true);
		dialog.setVisible(true);
	}

	/** Creates new form Spread */
	public Spread(java.awt.Frame parent, boolean modal)
	{
		super(parent, modal);
		initComponents();
        getRootPane().setDefaultButton(ok);
		spreadUp.setSelected(true);
	}

	protected void escapePressed() { cancel(null); }


	/** This method is called from within the constructor to
	 * initialize the form.
	 * WARNING: Do NOT modify this code. The content of this method is
	 * always regenerated by the Form Editor.
	 */
    private void initComponents()//GEN-BEGIN:initComponents
    {
        java.awt.GridBagConstraints gridBagConstraints;

        direction = new javax.swing.ButtonGroup();
        cancel = new javax.swing.JButton();
        ok = new javax.swing.JButton();
        jLabel1 = new javax.swing.JLabel();
        spreadAmount = new javax.swing.JTextField();
        spreadUp = new javax.swing.JRadioButton();
        spreadDown = new javax.swing.JRadioButton();
        spreadLeft = new javax.swing.JRadioButton();
        spreadRight = new javax.swing.JRadioButton();

        getContentPane().setLayout(new java.awt.GridBagLayout());

        setTitle("Spread About Highlighted");
        setName("");
        addWindowListener(new java.awt.event.WindowAdapter()
        {
            public void windowClosing(java.awt.event.WindowEvent evt)
            {
                closeDialog(evt);
            }
        });

        cancel.setText("Cancel");
        cancel.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                cancel(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.weightx = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(cancel, gridBagConstraints);

        ok.setText("OK");
        ok.addActionListener(new java.awt.event.ActionListener()
        {
            public void actionPerformed(java.awt.event.ActionEvent evt)
            {
                ok(evt);
            }
        });

        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 1;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.gridheight = 2;
        gridBagConstraints.weightx = 0.5;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(ok, gridBagConstraints);

        jLabel1.setText("Distance to spread:");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(jLabel1, gridBagConstraints);

        spreadAmount.setColumns(8);
        spreadAmount.setText(" ");
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 0;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.gridwidth = 2;
        gridBagConstraints.fill = java.awt.GridBagConstraints.HORIZONTAL;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(spreadAmount, gridBagConstraints);

        spreadUp.setText("Spread up");
        direction.add(spreadUp);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 0;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(spreadUp, gridBagConstraints);

        spreadDown.setText("Spread down");
        direction.add(spreadDown);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 1;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(spreadDown, gridBagConstraints);

        spreadLeft.setText("Spread left");
        direction.add(spreadLeft);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 2;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(spreadLeft, gridBagConstraints);

        spreadRight.setText("Spread right");
        direction.add(spreadRight);
        gridBagConstraints = new java.awt.GridBagConstraints();
        gridBagConstraints.gridx = 2;
        gridBagConstraints.gridy = 3;
        gridBagConstraints.anchor = java.awt.GridBagConstraints.WEST;
        gridBagConstraints.insets = new java.awt.Insets(4, 4, 4, 4);
        getContentPane().add(spreadRight, gridBagConstraints);

        pack();
    }//GEN-END:initComponents

	private void cancel(java.awt.event.ActionEvent evt)//GEN-FIRST:event_cancel
	{//GEN-HEADEREND:event_cancel
		closeDialog(null);
	}//GEN-LAST:event_cancel

	private void ok(java.awt.event.ActionEvent evt)//GEN-FIRST:event_ok
	{//GEN-HEADEREND:event_ok
		// spread it
		SpreadJob job = new SpreadJob(this);
		closeDialog(null);
	}//GEN-LAST:event_ok

	/** Closes the dialog */
	private void closeDialog(java.awt.event.WindowEvent evt)//GEN-FIRST:event_closeDialog
	{
		setVisible(false);
		dispose();
	}//GEN-LAST:event_closeDialog

	/**
	 * Class to create a cell in a new thread.
	 */
	private static class SpreadJob extends Job
	{
		Spread dialog;

		protected SpreadJob(Spread dialog)
		{
			super("Spread Circuitry", User.tool, Job.Type.CHANGE, null, null, Job.Priority.USER);
			this.dialog = dialog;
			startJob();
		}

		/**
		 * Method to implement the "spread" command.
		 * @param direction the direction to spread: 'u' for up, 'd' for down,
		 * 'l' for left, 'r' for right.
		 * @param amount the distance to spread (negative values compact).
		 */
		public boolean doIt()
		{
			// should ensure that the name is valid
			char direction = 0;
			if (dialog.spreadUp.isSelected()) direction = 'u'; else
			if (dialog.spreadDown.isSelected()) direction = 'd'; else
			if (dialog.spreadLeft.isSelected()) direction = 'l'; else
			if (dialog.spreadRight.isSelected()) direction = 'r';
			double amount = TextUtils.atof(dialog.spreadAmount.getText());

			NodeInst ni = (NodeInst)Highlight.getOneElectricObject(NodeInst.class);
			if (ni == null) return false;

			// disallow spreading if lock is on
			Cell cell = ni.getParent();
			if (CircuitChanges.cantEdit(cell, null, true)) return false;

			SizeOffset so = ni.getSizeOffset();
			double sLx = ni.getTrueCenterX() - ni.getXSize()/2 + so.getLowXOffset();
			double sHx = ni.getTrueCenterX() + ni.getXSize()/2 - so.getHighXOffset();
			double sLy = ni.getTrueCenterY() - ni.getYSize()/2 + so.getLowYOffset();
			double sHy = ni.getTrueCenterY() + ni.getYSize()/2 - so.getHighYOffset();

			// initialize by turning the marker bits off
			FlagSet fs = Geometric.getFlagSet(1);
			for(Iterator it = cell.getNodes(); it.hasNext(); )
			{
				NodeInst oNi = (NodeInst)it.next();
				oNi.clearBit(fs);
			}
			for(Iterator it = cell.getArcs(); it.hasNext(); )
			{
				ArcInst oAi = (ArcInst)it.next();
				oAi.clearBit(fs);
			}

			// set "already done" flag for nodes manhattan connected on spread line
			boolean mustBeHor = true;
			if (direction == 'l' || direction == 'r') mustBeHor = false;
			manhattanTravel(ni, mustBeHor, fs);

			// set "already done" flag for nodes that completely cover spread node or are in its line
			for(Iterator it = cell.getNodes(); it.hasNext(); )
			{
				NodeInst oNi = (NodeInst)it.next();
				SizeOffset oSo = oNi.getSizeOffset();
				if (direction == 'l' || direction == 'r')
				{
					if (oNi.getTrueCenterX() - oNi.getXSize()/2 + oSo.getLowXOffset() < sLx &&
						oNi.getTrueCenterX() + oNi.getXSize()/2 - oSo.getHighXOffset() > sHx)
							oNi.setBit(fs);
					if (oNi.getTrueCenterX() == (sLx+sHx)/2)
						oNi.setBit(fs);
				} else
				{
					if (oNi.getTrueCenterY() - oNi.getYSize()/2 + oSo.getLowYOffset() < sLy &&
						oNi.getTrueCenterY() + oNi.getYSize()/2 - oSo.getHighYOffset() > sHy)
							oNi.setBit(fs);
					if (oNi.getTrueCenterY() == (sLy+sHy)/2)
						oNi.setBit(fs);
				}
			}

			// mark those arcinsts that should stretch during spread
			for(Iterator it = cell.getArcs(); it.hasNext(); )
			{
				ArcInst ai = (ArcInst)it.next();
				NodeInst no1 = ai.getTail().getPortInst().getNodeInst();
				NodeInst no2 = ai.getHead().getPortInst().getNodeInst();
				double xC1 = no1.getTrueCenterX();
				double yC1 = no1.getTrueCenterY();
				double xC2 = no2.getTrueCenterX();
				double yC2 = no2.getTrueCenterY();

				// if one node is along spread line, make it "no1"
				if (no2.isBit(fs))
				{
					NodeInst swapNi = no1;  no1 = no2;  no2 = swapNi;
					double swap = xC1;     xC1 = xC2;  xC2 = swap;
					swap = yC1;     yC1 = yC2;  yC2 = swap;
				}

				// if both nodes are along spread line, leave arc alone
				if (no2.isBit(fs)) continue;

				boolean i = true;
				if (no1.isBit(fs))
				{
					// handle arcs connected to spread line
					switch (direction)
					{
						case 'l': if (xC2 <= sLx) i = false;   break;
						case 'r': if (xC2 >= sHx) i = false;   break;
						case 'u': if (yC2 >= sHy) i = false;   break;
						case 'd': if (yC2 <= sLy) i = false;   break;
					}
				} else
				{
					// handle arcs that cross the spread line
					switch (direction)
					{
						case 'l': if (xC1 > sLx && xC2 <= sLx) i = false; else
							if (xC2 > sLx && xC1 <= sLx) i = false;
							break;
						case 'r': if (xC1 < sHx && xC2 >= sHx) i = false; else
							if (xC2 < sHx && xC1 >= sHx) i = false;
							break;
						case 'u': if (yC1 > sHy && yC2 <= sHy) i = false; else
							if (yC2 > sHy && yC1 <= sHy) i = false;
							break;
						case 'd': if (yC1 < sLy && yC2 >= sLy) i = false; else
							if (yC2 < sLy && yC1 >= sLy) i = false;
							break;
					}
				}
				if (!i) ai.setBit(fs);
			}

			// now look at every nodeinst in the cell
			boolean moved = false;
			boolean again = true;
			while (again)
			{
				again = false;
				for(Iterator it = cell.getNodes(); it.hasNext(); )
				{
					NodeInst oNi = (NodeInst)it.next();

					// ignore this nodeinst if it has been spread already
					if (oNi.isBit(fs)) continue;

					// make sure nodeinst is on proper side of requested spread
					double xC1 = oNi.getTrueCenterX();
					double yC1 = oNi.getTrueCenterY();
					boolean doIt = false;
					switch (direction)
					{
						case 'l': if (xC1 < sLx) doIt = true;   break;
						case 'r': if (xC1 > sHx) doIt = true;   break;
						case 'u': if (yC1 > sHy) doIt = true;   break;
						case 'd': if (yC1 < sLy) doIt = true;   break;
					}
					if (!doIt) continue;

					// set every connecting nodeinst to be "spread"
					for(Iterator aIt = cell.getArcs(); aIt.hasNext(); )
					{
						ArcInst ai = (ArcInst)aIt.next();
						if (ai.isBit(fs))
						{
							// make arc temporarily unrigid
							Layout.setTempRigid(ai, false);
						} else
						{
							// make arc temporarily rigid
							Layout.setTempRigid(ai, true);
						}
					}
					netTravel(oNi, fs);

					// move this nodeinst in proper direction to do spread
					switch(direction)
					{
						case 'l':
							oNi.modifyInstance(-amount, 0, 0, 0, 0);
							break;
						case 'r':
							oNi.modifyInstance(amount, 0, 0, 0, 0);
							break;
						case 'u':
							oNi.modifyInstance(0, amount, 0, 0, 0);
							break;
						case 'd':
							oNi.modifyInstance(0, -amount, 0, 0, 0);
							break;
					}

					// set loop iteration flag and node spread flag
					moved = true;
					again = true;
					break;
				}
			}
			fs.freeFlagSet();
			if (!moved) System.out.println("Nothing changed");
			return true;
		}

		/**
		 * Method to travel through the network, setting flags.
		 * @param ni the NodeInst from which to start traveling.
		 * @param fs the FlagSet bit to mark during travel.
		 */
		private static void netTravel(NodeInst ni, FlagSet fs)
		{
			if (ni.isBit(fs)) return;
			ni.setBit(fs);

			for(Iterator it = ni.getConnections(); it.hasNext(); )
			{
				Connection con = (Connection)it.next();
				ArcInst ai = con.getArc();
				if (ai.isBit(fs)) continue;
				netTravel(ai.getHead().getPortInst().getNodeInst(), fs);
				netTravel(ai.getTail().getPortInst().getNodeInst(), fs);
			}
		}

		/**
		 * Method to recursively travel along all arcs on a NodeInst.
		 * @param ni the NodeInst to examine.
		 * @param hor true to travel along horizontal arcs; false for vertical.
		 * @param fs the FlagSet used to mark nodes that are examined.
		 * This is called from "spread" to propagate along manhattan
		 * arcs that are in the correct orientation (along the spread line).
		 */
		private static void manhattanTravel(NodeInst ni, boolean hor, FlagSet fs)
		{
			ni.setBit(fs);
			for(Iterator it = ni.getConnections(); it.hasNext(); )
			{
				Connection con = (Connection)it.next();
				ArcInst ai = con.getArc();
				int angle = ai.getAngle();
				if (hor)
				{
					// only want horizontal arcs
					if (angle != 0 && angle != 1800) continue;
				} else
				{
					// only want vertical arcs
					if (angle != 900 && angle != 2700) continue;
				}
				NodeInst other = null;
				if (ai.getHead() == con) other = ai.getTail().getPortInst().getNodeInst(); else
					other = ai.getHead().getPortInst().getNodeInst();
				if (other.isBit(fs)) continue;
				manhattanTravel(other, hor, fs);
			}
		}
	}

    // Variables declaration - do not modify//GEN-BEGIN:variables
    private javax.swing.JButton cancel;
    private javax.swing.ButtonGroup direction;
    private javax.swing.JLabel jLabel1;
    private javax.swing.JButton ok;
    private javax.swing.JTextField spreadAmount;
    private javax.swing.JRadioButton spreadDown;
    private javax.swing.JRadioButton spreadLeft;
    private javax.swing.JRadioButton spreadRight;
    private javax.swing.JRadioButton spreadUp;
    // End of variables declaration//GEN-END:variables
	
}
