/*
 * Decompiled with CFR 0.152.
 */
package org.wildfly.security.credential.store.impl;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.nio.file.LinkOption;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.Paths;
import java.security.AlgorithmParameters;
import java.security.GeneralSecurityException;
import java.security.InvalidKeyException;
import java.security.Key;
import java.security.KeyFactory;
import java.security.KeyPair;
import java.security.KeyStore;
import java.security.KeyStoreException;
import java.security.NoSuchAlgorithmException;
import java.security.PrivateKey;
import java.security.Provider;
import java.security.PublicKey;
import java.security.Security;
import java.security.UnrecoverableEntryException;
import java.security.cert.CertificateException;
import java.security.cert.CertificateFactory;
import java.security.cert.X509Certificate;
import java.security.spec.AlgorithmParameterSpec;
import java.security.spec.InvalidKeySpecException;
import java.security.spec.InvalidParameterSpecException;
import java.security.spec.PKCS8EncodedKeySpec;
import java.security.spec.X509EncodedKeySpec;
import java.util.Arrays;
import java.util.Collections;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.locks.ReadWriteLock;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.crypto.BadPaddingException;
import javax.crypto.Cipher;
import javax.crypto.NoSuchPaddingException;
import javax.crypto.SecretKey;
import javax.crypto.spec.IvParameterSpec;
import javax.crypto.spec.SecretKeySpec;
import org.wildfly.common.Assert;
import org.wildfly.security.EmptyProvider;
import org.wildfly.security._private.ElytronMessages;
import org.wildfly.security.asn1.ASN1Exception;
import org.wildfly.security.asn1.DERDecoder;
import org.wildfly.security.asn1.DEREncoder;
import org.wildfly.security.credential.AlgorithmCredential;
import org.wildfly.security.credential.BearerTokenCredential;
import org.wildfly.security.credential.Credential;
import org.wildfly.security.credential.KeyPairCredential;
import org.wildfly.security.credential.PasswordCredential;
import org.wildfly.security.credential.PublicKeyCredential;
import org.wildfly.security.credential.SecretKeyCredential;
import org.wildfly.security.credential.X509CertificateChainPrivateCredential;
import org.wildfly.security.credential.X509CertificateChainPublicCredential;
import org.wildfly.security.credential.source.CredentialSource;
import org.wildfly.security.credential.store.CredentialStore;
import org.wildfly.security.credential.store.CredentialStoreException;
import org.wildfly.security.credential.store.CredentialStoreSpi;
import org.wildfly.security.key.KeyUtil;
import org.wildfly.security.password.Password;
import org.wildfly.security.password.PasswordFactory;
import org.wildfly.security.password.interfaces.ClearPassword;
import org.wildfly.security.password.interfaces.MaskedPassword;
import org.wildfly.security.password.spec.ClearPasswordSpec;
import org.wildfly.security.password.spec.DigestPasswordSpec;
import org.wildfly.security.password.spec.HashPasswordSpec;
import org.wildfly.security.password.spec.IteratedSaltedHashPasswordSpec;
import org.wildfly.security.password.spec.MaskedPasswordSpec;
import org.wildfly.security.password.spec.OneTimePasswordSpec;
import org.wildfly.security.password.spec.PasswordSpec;
import org.wildfly.security.password.spec.SaltedHashPasswordSpec;
import org.wildfly.security.util.Alphabet;
import org.wildfly.security.util.AtomicFileOutputStream;
import org.wildfly.security.util.ByteIterator;
import org.wildfly.security.util.CodePointIterator;
import org.wildfly.security.x500.X500;

public final class KeyStoreCredentialStore
extends CredentialStoreSpi {
    private static final String DATA_OID = "1.2.840.113549.1.7.1";
    public static final String KEY_STORE_CREDENTIAL_STORE = KeyStoreCredentialStore.class.getSimpleName();
    private static final String X_509 = "X.509";
    private static final String CREATE = "create";
    private static final String CRYPTOALG = "cryptoAlg";
    private static final String EXTERNAL = "external";
    private static final String EXTERNALPATH = "externalPath";
    private static final String KEYALIAS = "keyAlias";
    private static final String KEYSTORETYPE = "keyStoreType";
    private static final String LOCATION = "location";
    private static final String MODIFIABLE = "modifiable";
    private static final List<String> validAttribtues = Arrays.asList("create", "cryptoAlg", "external", "externalPath", "keyAlias", "keyStoreType", "location", "modifiable");
    private final ReadWriteLock readWriteLock = new ReentrantReadWriteLock();
    private final HashMap<String, TopEntry> cache = new HashMap();
    private volatile boolean modifiable;
    private KeyStore keyStore;
    private Path location;
    private Path externalPath;
    private boolean create;
    private CredentialStore.ProtectionParameter protectionParameter;
    private Provider[] providers;
    private String encryptionKeyAlias;
    private boolean useExternalStorage = false;
    private ExternalStorage externalStorage;
    private String cryptographicAlgorithm;
    private static final Pattern INDEX_PATTERN = Pattern.compile("(.+)/([a-z0-9_]+)/([-a-z0-9_]+)?/([2-7a-z]+)?$");
    private static final Map<String, Class<? extends Credential>> CREDENTIAL_TYPES;

    @Override
    public void initialize(Map<String, String> attributes, CredentialStore.ProtectionParameter protectionParameter, Provider[] providers) throws CredentialStoreException {
        try (Hold hold = this.lockForWrite();){
            if (protectionParameter == null) {
                throw ElytronMessages.log.protectionParameterRequired();
            }
            this.validateAttribute(attributes, validAttribtues);
            this.cache.clear();
            this.protectionParameter = protectionParameter;
            this.modifiable = Boolean.parseBoolean(attributes.getOrDefault(MODIFIABLE, "true"));
            this.create = Boolean.parseBoolean(attributes.getOrDefault(CREATE, "false"));
            String locationName = attributes.get(LOCATION);
            this.location = locationName == null ? null : Paths.get(locationName, new String[0]);
            this.providers = providers;
            String keyStoreType = attributes.getOrDefault(KEYSTORETYPE, KeyStore.getDefaultType());
            this.useExternalStorage = Boolean.parseBoolean(attributes.getOrDefault(EXTERNAL, "false"));
            if (this.useExternalStorage) {
                String externalPathName = attributes.get(EXTERNALPATH);
                if (externalPathName == null) {
                    throw ElytronMessages.log.externalPathMissing(keyStoreType);
                }
                this.externalPath = Paths.get(externalPathName, new String[0]);
                if (this.externalPath.equals(this.location)) {
                    throw ElytronMessages.log.locationAndExternalPathAreIdentical(this.location.toString(), this.externalPath.toString());
                }
            }
            this.encryptionKeyAlias = attributes.getOrDefault(KEYALIAS, "cs_key");
            this.cryptographicAlgorithm = attributes.get(CRYPTOALG);
            this.load(keyStoreType);
            if (this.create && !this.useExternalStorage && this.location != null && !Files.exists(this.location, new LinkOption[0])) {
                this.flush();
            }
            this.initialized = true;
        }
    }

    @Override
    public boolean isModifiable() {
        return this.modifiable;
    }

    @Override
    public void store(String credentialAlias, Credential credential, CredentialStore.ProtectionParameter protectionParameter) throws CredentialStoreException {
        try {
            KeyStore.Entry entry;
            Class<?> credentialClass = credential.getClass();
            String algorithmName = credential instanceof AlgorithmCredential ? ((AlgorithmCredential)credential).getAlgorithm() : null;
            AlgorithmParameterSpec parameterSpec = credential.castAndApply(AlgorithmCredential.class, AlgorithmCredential::getParameters);
            if (credentialClass == SecretKeyCredential.class) {
                entry = new KeyStore.SecretKeyEntry(credential.castAndApply(SecretKeyCredential.class, SecretKeyCredential::getSecretKey));
            } else if (credentialClass == PublicKeyCredential.class) {
                PublicKey publicKey = credential.castAndApply(PublicKeyCredential.class, PublicKeyCredential::getPublicKey);
                KeyFactory keyFactory = KeyFactory.getInstance(publicKey.getAlgorithm());
                X509EncodedKeySpec keySpec = keyFactory.getKeySpec(keyFactory.translateKey(publicKey), X509EncodedKeySpec.class);
                byte[] encoded = keySpec.getEncoded();
                entry = new KeyStore.SecretKeyEntry(new SecretKeySpec(encoded, DATA_OID));
            } else if (credentialClass == KeyPairCredential.class) {
                KeyPair keyPair = credential.castAndApply(KeyPairCredential.class, KeyPairCredential::getKeyPair);
                PublicKey publicKey = keyPair.getPublic();
                X509Certificate[] privateKey = keyPair.getPrivate();
                KeyFactory keyFactory = KeyFactory.getInstance(publicKey.getAlgorithm());
                assert (privateKey.getAlgorithm().equals(publicKey.getAlgorithm()));
                X509EncodedKeySpec publicSpec = keyFactory.getKeySpec(keyFactory.translateKey(publicKey), X509EncodedKeySpec.class);
                PKCS8EncodedKeySpec privateSpec = keyFactory.getKeySpec(keyFactory.translateKey((Key)privateKey), PKCS8EncodedKeySpec.class);
                DEREncoder encoder = new DEREncoder();
                encoder.startSequence();
                encoder.writeEncoded(publicSpec.getEncoded());
                encoder.writeEncoded(privateSpec.getEncoded());
                encoder.endSequence();
                entry = new KeyStore.SecretKeyEntry(new SecretKeySpec(encoder.getEncoded(), DATA_OID));
            } else if (credentialClass == X509CertificateChainPublicCredential.class) {
                X509Certificate[] x509Certificates = credential.castAndApply(X509CertificateChainPublicCredential.class, rec$ -> ((X509CertificateChainPublicCredential)rec$).getCertificateChain());
                DEREncoder encoder = new DEREncoder();
                encoder.encodeInteger(x509Certificates.length);
                encoder.startSequence();
                for (X509Certificate x509Certificate : x509Certificates) {
                    encoder.writeEncoded(x509Certificate.getEncoded());
                }
                encoder.endSequence();
                entry = new KeyStore.SecretKeyEntry(new SecretKeySpec(encoder.getEncoded(), DATA_OID));
            } else if (credentialClass == X509CertificateChainPrivateCredential.class) {
                X509CertificateChainPrivateCredential cred = (X509CertificateChainPrivateCredential)credential;
                entry = new KeyStore.PrivateKeyEntry(cred.getPrivateKey(), cred.getCertificateChain());
            } else if (credentialClass == BearerTokenCredential.class) {
                entry = new KeyStore.SecretKeyEntry(new SecretKeySpec(credential.castAndApply(BearerTokenCredential.class, c -> c.getToken().getBytes(StandardCharsets.UTF_8)), DATA_OID));
            } else if (credentialClass == PasswordCredential.class) {
                Password password = credential.castAndApply(PasswordCredential.class, PasswordCredential::getPassword);
                String algorithm = password.getAlgorithm();
                DEREncoder encoder = new DEREncoder();
                PasswordFactory passwordFactory = PasswordFactory.getInstance(algorithm);
                switch (algorithm) {
                    case "bcrypt": 
                    case "bsd-crypt-des": 
                    case "scram-sha-1": 
                    case "scram-sha-256": 
                    case "scram-sha-384": 
                    case "scram-sha-512": 
                    case "sun-crypt-md5": 
                    case "sun-crypt-md5-bare-salt": 
                    case "crypt-sha-256": 
                    case "crypt-sha-512": {
                        PasswordSpec passwordSpec = passwordFactory.getKeySpec(passwordFactory.translate(password), IteratedSaltedHashPasswordSpec.class);
                        encoder.startSequence();
                        encoder.encodeOctetString(((IteratedSaltedHashPasswordSpec)passwordSpec).getHash());
                        encoder.encodeOctetString(((IteratedSaltedHashPasswordSpec)passwordSpec).getSalt());
                        encoder.encodeInteger(((IteratedSaltedHashPasswordSpec)passwordSpec).getIterationCount());
                        encoder.endSequence();
                        break;
                    }
                    case "clear": {
                        PasswordSpec passwordSpec = passwordFactory.getKeySpec(passwordFactory.translate(password), ClearPasswordSpec.class);
                        encoder.encodeOctetString(new String(((ClearPasswordSpec)passwordSpec).getEncodedPassword()));
                        break;
                    }
                    case "digest-md5": 
                    case "digest-sha": 
                    case "digest-sha-256": 
                    case "digest-sha-384": 
                    case "digest-sha-512": 
                    case "digest-sha-512-256": {
                        PasswordSpec passwordSpec = passwordFactory.getKeySpec(passwordFactory.translate(password), DigestPasswordSpec.class);
                        encoder.startSequence();
                        encoder.encodeOctetString(((DigestPasswordSpec)passwordSpec).getUsername());
                        encoder.encodeOctetString(((DigestPasswordSpec)passwordSpec).getRealm());
                        encoder.encodeOctetString(((DigestPasswordSpec)passwordSpec).getDigest());
                        encoder.endSequence();
                        break;
                    }
                    case "otp-md5": 
                    case "otp-sha1": 
                    case "otp-sha256": 
                    case "otp-sha384": 
                    case "otp-sha512": {
                        PasswordSpec passwordSpec = passwordFactory.getKeySpec(passwordFactory.translate(password), OneTimePasswordSpec.class);
                        encoder.startSequence();
                        encoder.encodeOctetString(((OneTimePasswordSpec)passwordSpec).getHash());
                        encoder.encodeIA5String(((OneTimePasswordSpec)passwordSpec).getSeed());
                        encoder.encodeInteger(((OneTimePasswordSpec)passwordSpec).getSequenceNumber());
                        encoder.endSequence();
                        break;
                    }
                    case "password-salt-digest-md5": 
                    case "password-salt-digest-sha-1": 
                    case "password-salt-digest-sha-256": 
                    case "password-salt-digest-sha-384": 
                    case "password-salt-digest-sha-512": 
                    case "salt-password-digest-md5": 
                    case "salt-password-digest-sha-1": 
                    case "salt-password-digest-sha-256": 
                    case "salt-password-digest-sha-384": 
                    case "salt-password-digest-sha-512": 
                    case "crypt-des": 
                    case "crypt-md5": {
                        PasswordSpec passwordSpec = passwordFactory.getKeySpec(passwordFactory.translate(password), SaltedHashPasswordSpec.class);
                        encoder.startSequence();
                        encoder.encodeOctetString(((SaltedHashPasswordSpec)passwordSpec).getHash());
                        encoder.encodeOctetString(((SaltedHashPasswordSpec)passwordSpec).getSalt());
                        encoder.endSequence();
                        break;
                    }
                    case "simple-digest-md2": 
                    case "simple-digest-md5": 
                    case "simple-digest-sha-1": 
                    case "simple-digest-sha-256": 
                    case "simple-digest-sha-384": 
                    case "simple-digest-sha-512": {
                        PasswordSpec passwordSpec = passwordFactory.getKeySpec(passwordFactory.translate(password), HashPasswordSpec.class);
                        encoder.startSequence();
                        encoder.encodeOctetString(((HashPasswordSpec)passwordSpec).getDigest());
                        encoder.endSequence();
                        break;
                    }
                    default: {
                        PasswordSpec passwordSpec;
                        if (MaskedPassword.isMaskedAlgorithm(algorithmName)) {
                            passwordSpec = passwordFactory.getKeySpec(passwordFactory.translate(password), MaskedPasswordSpec.class);
                            encoder.startSequence();
                            encoder.encodeOctetString(new String(((MaskedPasswordSpec)passwordSpec).getInitialKeyMaterial()));
                            encoder.encodeInteger(((MaskedPasswordSpec)passwordSpec).getIterationCount());
                            encoder.encodeOctetString(((MaskedPasswordSpec)passwordSpec).getSalt());
                            encoder.encodeOctetString(((MaskedPasswordSpec)passwordSpec).getMaskedPasswordBytes());
                            encoder.endSequence();
                            break;
                        }
                        throw ElytronMessages.log.unsupportedCredentialType(credentialClass);
                    }
                }
                entry = new KeyStore.SecretKeyEntry(new SecretKeySpec(encoder.getEncoded(), DATA_OID));
            } else {
                throw ElytronMessages.log.unsupportedCredentialType(credentialClass);
            }
            String ksAlias = this.calculateNewAlias(credentialAlias, credentialClass, algorithmName, parameterSpec);
            try (Hold hold = this.lockForWrite();){
                this.keyStore.setEntry(ksAlias, entry, this.convertParameter(protectionParameter));
                TopEntry topEntry = this.cache.computeIfAbsent(KeyStoreCredentialStore.toLowercase(credentialAlias), TopEntry::new);
                MidEntry midEntry = topEntry.getMap().computeIfAbsent(credentialClass, c -> new MidEntry(topEntry, (Class<? extends Credential>)c));
                BottomEntry bottomEntry = algorithmName != null ? midEntry.getMap().computeIfAbsent(algorithmName, n -> new BottomEntry(midEntry, (String)n)) : midEntry.getOrCreateNoAlgorithm();
                String oldAlias = parameterSpec != null ? bottomEntry.getMap().put(new ParamKey(parameterSpec), ksAlias) : bottomEntry.setNoParams(ksAlias);
                if (oldAlias != null && !oldAlias.equals(ksAlias)) {
                    this.keyStore.deleteEntry(oldAlias);
                }
            }
        }
        catch (InvalidKeyException | KeyStoreException | NoSuchAlgorithmException | CertificateException | InvalidKeySpecException e) {
            throw ElytronMessages.log.cannotWriteCredentialToStore(e);
        }
    }

    /*
     * Enabled aggressive block sorting
     * Enabled unnecessary exception pruning
     * Enabled aggressive exception aggregation
     */
    @Override
    public <C extends Credential> C retrieve(String credentialAlias, Class<C> credentialType, String credentialAlgorithm, AlgorithmParameterSpec parameterSpec, CredentialStore.ProtectionParameter protectionParameter) throws CredentialStoreException {
        DERDecoder decoder;
        String matchedAlgorithm;
        byte[] encoded;
        SecretKey secretKey;
        KeyStore.Entry entry;
        BottomEntry bottomEntry;
        MidEntry midEntry;
        try (Hold hold = this.lockForRead();){
            Iterator<Object> iterator;
            TopEntry topEntry = this.cache.get(KeyStoreCredentialStore.toLowercase(credentialAlias));
            if (topEntry == null) {
                ElytronMessages.log.trace("KeyStoreCredentialStore: alias not found in cache");
                C c = null;
                return c;
            }
            if (topEntry.getMap().containsKey(credentialType)) {
                ElytronMessages.log.trace("KeyStoreCredentialStore: contains exact type");
                midEntry = topEntry.getMap().get(credentialType);
            } else {
                MidEntry item;
                iterator = topEntry.getMap().values().iterator();
                do {
                    if (iterator.hasNext()) continue;
                    ElytronMessages.log.trace("KeyStoreCredentialStore: no assignable found");
                    C c = null;
                    return c;
                } while (!credentialType.isAssignableFrom((item = iterator.next()).getCredentialType()));
                ElytronMessages.log.trace("KeyStoreCredentialStore: assignable found");
                midEntry = item;
            }
            bottomEntry = credentialAlgorithm != null ? midEntry.getMap().get(credentialAlgorithm) : ((iterator = midEntry.getMap().values().iterator()).hasNext() ? (BottomEntry)iterator.next() : midEntry.getNoAlgorithm());
            if (bottomEntry == null) {
                ElytronMessages.log.tracef("KeyStoreCredentialStore: no entry for algorithm %s", (Object)credentialAlgorithm);
                iterator = null;
                return (C)iterator;
            }
            String ksAlias = parameterSpec != null ? bottomEntry.getMap().get(new ParamKey(parameterSpec)) : ((iterator = bottomEntry.getMap().values().iterator()).hasNext() ? (String)iterator.next() : bottomEntry.getNoParams());
            if (ksAlias == null) {
                ElytronMessages.log.tracef("KeyStoreCredentialStore: no entry for parameterSpec %s", (Object)parameterSpec);
                iterator = null;
                return (C)iterator;
            }
            entry = this.keyStore.getEntry(ksAlias, this.convertParameter(protectionParameter));
        }
        catch (KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException e) {
            throw ElytronMessages.log.cannotAcquireCredentialFromStore(e);
        }
        if (entry == null) {
            ElytronMessages.log.trace("KeyStoreCredentialStore: null entry");
            return null;
        }
        Class<? extends Credential> matchedCredentialType = midEntry.getCredentialType();
        if (matchedCredentialType == SecretKeyCredential.class) {
            if (!(entry instanceof KeyStore.SecretKeyEntry)) throw ElytronMessages.log.invalidCredentialStoreEntryType(KeyStore.SecretKeyEntry.class, entry.getClass());
            secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
            return (C)((Credential)credentialType.cast(new SecretKeyCredential(secretKey)));
        }
        if (matchedCredentialType == PublicKeyCredential.class) {
            if (!(entry instanceof KeyStore.SecretKeyEntry)) throw ElytronMessages.log.invalidCredentialStoreEntryType(KeyStore.SecretKeyEntry.class, entry.getClass());
            try {
                secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
                encoded = secretKey.getEncoded();
                matchedAlgorithm = bottomEntry.getAlgorithm();
                assert (matchedAlgorithm != null);
                KeyFactory keyFactory = KeyFactory.getInstance(matchedAlgorithm);
                PublicKey publicKey = keyFactory.generatePublic(new X509EncodedKeySpec(encoded));
                return (C)((Credential)credentialType.cast(new PublicKeyCredential(publicKey)));
            }
            catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
                throw ElytronMessages.log.cannotAcquireCredentialFromStore(e);
            }
        }
        if (matchedCredentialType == KeyPairCredential.class) {
            if (!(entry instanceof KeyStore.SecretKeyEntry)) throw ElytronMessages.log.invalidCredentialStoreEntryType(KeyStore.SecretKeyEntry.class, entry.getClass());
            try {
                secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
                encoded = secretKey.getEncoded();
                matchedAlgorithm = bottomEntry.getAlgorithm();
                assert (matchedAlgorithm != null);
                decoder = new DERDecoder(encoded);
                decoder.startSequence();
                byte[] publicBytes = decoder.drainElement();
                byte[] privateBytes = decoder.drainElement();
                decoder.endSequence();
                KeyFactory keyFactory = KeyFactory.getInstance(matchedAlgorithm);
                PublicKey publicKey = keyFactory.generatePublic(new X509EncodedKeySpec(publicBytes));
                PrivateKey privateKey = keyFactory.generatePrivate(new PKCS8EncodedKeySpec(privateBytes));
                KeyPair keyPair = new KeyPair(publicKey, privateKey);
                return (C)((Credential)credentialType.cast(new KeyPairCredential(keyPair)));
            }
            catch (NoSuchAlgorithmException | InvalidKeySpecException | ASN1Exception e) {
                throw ElytronMessages.log.cannotAcquireCredentialFromStore(e);
            }
        }
        if (matchedCredentialType == X509CertificateChainPublicCredential.class) {
            if (!(entry instanceof KeyStore.SecretKeyEntry)) throw ElytronMessages.log.invalidCredentialStoreEntryType(KeyStore.SecretKeyEntry.class, entry.getClass());
            try {
                secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
                encoded = secretKey.getEncoded();
                matchedAlgorithm = bottomEntry.getAlgorithm();
                assert (matchedAlgorithm != null);
                decoder = new DERDecoder(encoded);
                CertificateFactory certificateFactory = CertificateFactory.getInstance(X_509);
                int count = decoder.decodeInteger().intValueExact();
                X509Certificate[] array = new X509Certificate[count];
                decoder.startSequence();
                int i = 0;
                while (true) {
                    if (!decoder.hasNextElement()) {
                        decoder.endSequence();
                        return (C)((Credential)credentialType.cast(new X509CertificateChainPublicCredential(array)));
                    }
                    byte[] certBytes = decoder.drainElement();
                    array[i++] = (X509Certificate)certificateFactory.generateCertificate(new ByteArrayInputStream(certBytes));
                }
            }
            catch (ArrayIndexOutOfBoundsException | CertificateException | ASN1Exception e) {
                throw ElytronMessages.log.cannotAcquireCredentialFromStore(e);
            }
        }
        if (matchedCredentialType == X509CertificateChainPrivateCredential.class) {
            if (!(entry instanceof KeyStore.PrivateKeyEntry)) throw ElytronMessages.log.invalidCredentialStoreEntryType(KeyStore.PrivateKeyEntry.class, entry.getClass());
            KeyStore.PrivateKeyEntry privateKeyEntry = (KeyStore.PrivateKeyEntry)entry;
            PrivateKey privateKey = privateKeyEntry.getPrivateKey();
            Object[] certificateChain = privateKeyEntry.getCertificateChain();
            X509Certificate[] x509Certificates = X500.asX509CertificateArray(certificateChain);
            return (C)((Credential)credentialType.cast(new X509CertificateChainPrivateCredential(privateKey, x509Certificates)));
        }
        if (matchedCredentialType == BearerTokenCredential.class) {
            if (!(entry instanceof KeyStore.SecretKeyEntry)) throw ElytronMessages.log.invalidCredentialStoreEntryType(KeyStore.SecretKeyEntry.class, entry.getClass());
            secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
            encoded = secretKey.getEncoded();
            return (C)((Credential)credentialType.cast(new BearerTokenCredential(new String(encoded, StandardCharsets.UTF_8))));
        }
        if (matchedCredentialType != PasswordCredential.class) throw ElytronMessages.log.unableToReadCredentialTypeFromStore(matchedCredentialType);
        if (!(entry instanceof KeyStore.SecretKeyEntry)) throw ElytronMessages.log.invalidCredentialStoreEntryType(KeyStore.SecretKeyEntry.class, entry.getClass());
        try {
            PasswordSpec passwordSpec;
            secretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
            encoded = secretKey.getEncoded();
            matchedAlgorithm = bottomEntry.getAlgorithm();
            assert (matchedAlgorithm != null);
            decoder = new DERDecoder(encoded);
            switch (matchedAlgorithm) {
                case "bcrypt": 
                case "bsd-crypt-des": 
                case "scram-sha-1": 
                case "scram-sha-256": 
                case "scram-sha-384": 
                case "scram-sha-512": 
                case "sun-crypt-md5": 
                case "sun-crypt-md5-bare-salt": 
                case "crypt-sha-256": 
                case "crypt-sha-512": {
                    decoder.startSequence();
                    byte[] hash = decoder.decodeOctetString();
                    byte[] salt = decoder.decodeOctetString();
                    int iterationCount = decoder.decodeInteger().intValue();
                    decoder.endSequence();
                    passwordSpec = new IteratedSaltedHashPasswordSpec(hash, salt, iterationCount);
                    break;
                }
                case "clear": {
                    passwordSpec = new ClearPasswordSpec(decoder.decodeOctetStringAsString().toCharArray());
                    break;
                }
                case "digest-md5": 
                case "digest-sha": 
                case "digest-sha-256": 
                case "digest-sha-384": 
                case "digest-sha-512": 
                case "digest-sha-512-256": {
                    decoder.startSequence();
                    String username = decoder.decodeOctetStringAsString();
                    String realm = decoder.decodeOctetStringAsString();
                    byte[] digest = decoder.decodeOctetString();
                    decoder.endSequence();
                    passwordSpec = new DigestPasswordSpec(username, realm, digest);
                    break;
                }
                case "otp-md5": 
                case "otp-sha1": 
                case "otp-sha256": 
                case "otp-sha384": 
                case "otp-sha512": {
                    decoder.startSequence();
                    byte[] hash = decoder.decodeOctetString();
                    String seed = decoder.decodeIA5String();
                    int sequenceNumber = decoder.decodeInteger().intValue();
                    decoder.endSequence();
                    passwordSpec = new OneTimePasswordSpec(hash, seed, sequenceNumber);
                    break;
                }
                case "password-salt-digest-md5": 
                case "password-salt-digest-sha-1": 
                case "password-salt-digest-sha-256": 
                case "password-salt-digest-sha-384": 
                case "password-salt-digest-sha-512": 
                case "salt-password-digest-md5": 
                case "salt-password-digest-sha-1": 
                case "salt-password-digest-sha-256": 
                case "salt-password-digest-sha-384": 
                case "salt-password-digest-sha-512": 
                case "crypt-des": 
                case "crypt-md5": {
                    decoder.startSequence();
                    byte[] hash = decoder.decodeOctetString();
                    byte[] salt = decoder.decodeOctetString();
                    decoder.endSequence();
                    passwordSpec = new SaltedHashPasswordSpec(hash, salt);
                    break;
                }
                case "simple-digest-md2": 
                case "simple-digest-md5": 
                case "simple-digest-sha-1": 
                case "simple-digest-sha-256": 
                case "simple-digest-sha-384": 
                case "simple-digest-sha-512": {
                    decoder.startSequence();
                    byte[] hash = decoder.decodeOctetString();
                    decoder.endSequence();
                    passwordSpec = new HashPasswordSpec(hash);
                    break;
                }
                default: {
                    if (!MaskedPassword.isMaskedAlgorithm(matchedAlgorithm)) throw ElytronMessages.log.unsupportedCredentialType(credentialType);
                    decoder.startSequence();
                    char[] initialKeyMaterial = decoder.decodeOctetStringAsString().toCharArray();
                    int iterationCount = decoder.decodeInteger().intValue();
                    byte[] salt = decoder.decodeOctetString();
                    byte[] maskedPasswordBytes = decoder.decodeOctetString();
                    decoder.endSequence();
                    passwordSpec = new MaskedPasswordSpec(initialKeyMaterial, iterationCount, salt, maskedPasswordBytes);
                    break;
                }
            }
            PasswordFactory passwordFactory = PasswordFactory.getInstance(matchedAlgorithm);
            Password password = passwordFactory.generatePassword(passwordSpec);
            return (C)((Credential)credentialType.cast(new PasswordCredential(password)));
        }
        catch (NoSuchAlgorithmException | InvalidKeySpecException e) {
            throw ElytronMessages.log.cannotAcquireCredentialFromStore(e);
        }
    }

    private KeyStore.ProtectionParameter convertParameter(CredentialStore.ProtectionParameter protectionParameter) throws CredentialStoreException {
        if (protectionParameter == null) {
            return this.convertParameter(this.protectionParameter);
        }
        if (protectionParameter instanceof CredentialStore.CredentialSourceProtectionParameter) {
            CredentialSource credentialSource = ((CredentialStore.CredentialSourceProtectionParameter)protectionParameter).getCredentialSource();
            try {
                return credentialSource.applyToCredential(PasswordCredential.class, c -> c.getPassword().castAndApply(ClearPassword.class, p -> new KeyStore.PasswordProtection(p.getPassword())));
            }
            catch (IOException e) {
                throw ElytronMessages.log.cannotAcquireCredentialFromStore(e);
            }
        }
        throw ElytronMessages.log.invalidProtectionParameter(protectionParameter);
    }

    @Override
    public void remove(String credentialAlias, Class<? extends Credential> credentialType, String credentialAlgorithm, AlgorithmParameterSpec parameterSpec) throws CredentialStoreException {
        String credentialAliasLowerCase = KeyStoreCredentialStore.toLowercase(credentialAlias);
        try (Hold hold = this.lockForWrite();){
            if (!this.modifiable) {
                throw ElytronMessages.log.nonModifiableCredentialStore("remove");
            }
            TopEntry topEntry = this.cache.get(credentialAliasLowerCase);
            if (topEntry == null) {
                return;
            }
            if (topEntry.getMap().containsKey(credentialType)) {
                this.remove(topEntry.getMap().remove(credentialType), credentialAlgorithm, parameterSpec);
            } else {
                Iterator<MidEntry> iterator = topEntry.getMap().values().iterator();
                while (iterator.hasNext()) {
                    MidEntry item = iterator.next();
                    if (!credentialType.isAssignableFrom(item.getCredentialType())) continue;
                    this.remove(item, credentialAlgorithm, parameterSpec);
                    if (!item.isEmpty()) continue;
                    iterator.remove();
                }
            }
            this.cache.remove(credentialAliasLowerCase);
        }
        catch (KeyStoreException e) {
            throw ElytronMessages.log.cannotRemoveCredentialFromStore(e);
        }
    }

    private void remove(MidEntry midEntry, String credentialAlgorithm, AlgorithmParameterSpec parameterSpec) throws KeyStoreException {
        if (midEntry != null) {
            if (credentialAlgorithm != null) {
                this.remove(midEntry.getMap().get(credentialAlgorithm), parameterSpec);
            } else {
                Iterator<BottomEntry> iterator = midEntry.getMap().values().iterator();
                while (iterator.hasNext()) {
                    BottomEntry item = iterator.next();
                    this.remove(item, parameterSpec);
                    if (!item.isEmpty()) continue;
                    iterator.remove();
                }
                this.remove(midEntry.removeNoAlgorithm(), parameterSpec);
            }
        }
    }

    private void remove(BottomEntry bottomEntry, AlgorithmParameterSpec parameterSpec) throws KeyStoreException {
        if (bottomEntry != null) {
            if (parameterSpec != null) {
                this.remove(bottomEntry.getMap().remove(new ParamKey(parameterSpec)));
            } else {
                Iterator<String> iterator = bottomEntry.getMap().values().iterator();
                while (iterator.hasNext()) {
                    String item = iterator.next();
                    this.remove(item);
                    iterator.remove();
                }
                this.remove(bottomEntry.removeNoParams());
            }
        }
    }

    private void remove(String ksAlias) throws KeyStoreException {
        if (ksAlias != null) {
            this.keyStore.deleteEntry(ksAlias);
        }
    }

    @Override
    public void flush() throws CredentialStoreException {
        block32: {
            try (Hold hold = this.lockForWrite();){
                Path dataLocation = this.externalPath == null ? this.location : this.externalPath;
                ElytronMessages.log.tracef("KeyStoreCredentialStore: flushing into %s", (Object)dataLocation);
                if (dataLocation == null) break block32;
                try {
                    char[] storePassword = KeyStoreCredentialStore.getStorePassword(this.protectionParameter);
                    try (AtomicFileOutputStream os = new AtomicFileOutputStream(dataLocation);){
                        try {
                            if (this.useExternalStorage) {
                                this.externalStorage.store(os);
                            } else {
                                this.keyStore.store(os, storePassword);
                            }
                        }
                        catch (Throwable t) {
                            try {
                                os.cancel();
                            }
                            catch (IOException e) {
                                e.addSuppressed(t);
                                throw e;
                            }
                        }
                    }
                }
                catch (IOException e) {
                    throw ElytronMessages.log.cannotFlushCredentialStore(e);
                }
            }
        }
    }

    @Override
    public Set<String> getAliases() throws UnsupportedOperationException, CredentialStoreException {
        return this.cache.keySet();
    }

    private Hold lockForRead() {
        this.readWriteLock.readLock().lock();
        return () -> this.readWriteLock.readLock().unlock();
    }

    private Hold lockForWrite() {
        this.readWriteLock.writeLock().lock();
        return () -> this.readWriteLock.writeLock().unlock();
    }

    private void load(String type) throws CredentialStoreException {
        Enumeration<Object> enumeration;
        block34: {
            Path dataLocation;
            if (this.useExternalStorage) {
                dataLocation = this.externalPath;
                this.setupExternalStorage(type, this.location);
            } else {
                dataLocation = this.location;
                this.keyStore = this.getKeyStoreInstance(type);
            }
            if (this.create) {
                ElytronMessages.log.tracef("KeyStoreCredentialStore: creating empty backing KeyStore  dataLocation = %s  external = %b", (Object)dataLocation, (Object)this.useExternalStorage);
                if (dataLocation == null) {
                    try {
                        this.keyStore.load(null, null);
                    }
                    catch (IOException | NoSuchAlgorithmException | CertificateException e) {
                        throw ElytronMessages.log.cannotInitializeCredentialStore(e);
                    }
                }
            } else if (dataLocation != null && !Files.exists(dataLocation, new LinkOption[0])) {
                throw ElytronMessages.log.automaticStorageCreationDisabled(dataLocation.toString());
            }
            try {
                if (dataLocation != null && Files.exists(dataLocation, new LinkOption[0])) {
                    ElytronMessages.log.tracef("KeyStoreCredentialStore: loading backing KeyStore %s  external = %b", (Object)dataLocation, (Object)this.useExternalStorage);
                    char[] password = KeyStoreCredentialStore.getStorePassword(this.protectionParameter);
                    try (InputStream fileStream = Files.newInputStream(dataLocation, new OpenOption[0]);){
                        if (this.useExternalStorage) {
                            this.externalStorage.load(fileStream);
                        } else {
                            this.keyStore.load(fileStream, password);
                        }
                    }
                    enumeration = this.keyStore.aliases();
                    break block34;
                }
                this.keyStore.load(null, null);
                enumeration = Collections.emptyEnumeration();
            }
            catch (GeneralSecurityException e) {
                throw ElytronMessages.log.cannotInitializeCredentialStore(ElytronMessages.log.internalEncryptionProblem(e, dataLocation != null ? dataLocation.toString() : "null"));
            }
            catch (IOException e) {
                throw ElytronMessages.log.cannotInitializeCredentialStore(e);
            }
        }
        while (enumeration.hasMoreElements()) {
            String ksAlias = ((String)enumeration.nextElement()).toLowerCase(Locale.ROOT);
            try {
                Matcher matcher = INDEX_PATTERN.matcher(ksAlias);
                if (matcher.matches()) {
                    BottomEntry bottomEntry;
                    MidEntry midEntry;
                    TopEntry topEntry;
                    String alias = matcher.group(1);
                    String credTypeName = matcher.group(2);
                    String algName = matcher.group(3);
                    String parameters = matcher.group(4);
                    Class<? extends Credential> credentialType = CREDENTIAL_TYPES.get(credTypeName);
                    if (credentialType == null) {
                        ElytronMessages.log.logIgnoredUnrecognizedKeyStoreEntry(ksAlias);
                        continue;
                    }
                    if (algName != null) {
                        if (parameters != null) {
                            byte[] encodedParameters = CodePointIterator.ofString(parameters).base32Decode(Alphabet.Base32Alphabet.LOWERCASE, false).drain();
                            AlgorithmParameters algorithmParameters = AlgorithmParameters.getInstance(algName);
                            algorithmParameters.init(encodedParameters);
                            AlgorithmParameterSpec parameterSpec = algorithmParameters.getParameterSpec(AlgorithmParameterSpec.class);
                            TopEntry topEntry2 = this.cache.computeIfAbsent(alias, TopEntry::new);
                            MidEntry midEntry2 = topEntry2.getMap().computeIfAbsent(credentialType, k -> new MidEntry(topEntry2, (Class<? extends Credential>)k));
                            BottomEntry bottomEntry2 = midEntry2.getMap().computeIfAbsent(algName, k -> new BottomEntry(midEntry2, (String)k));
                            bottomEntry2.getMap().put(new ParamKey(parameterSpec), ksAlias);
                            continue;
                        }
                        topEntry = this.cache.computeIfAbsent(alias, TopEntry::new);
                        midEntry = topEntry.getMap().computeIfAbsent(credentialType, k -> new MidEntry(topEntry, (Class<? extends Credential>)k));
                        bottomEntry = midEntry.getMap().computeIfAbsent(algName, k -> new BottomEntry(midEntry, (String)k));
                        bottomEntry.setNoParams(ksAlias);
                        continue;
                    }
                    topEntry = this.cache.computeIfAbsent(alias, TopEntry::new);
                    midEntry = topEntry.getMap().computeIfAbsent(credentialType, k -> new MidEntry(topEntry, (Class<? extends Credential>)k));
                    bottomEntry = midEntry.getOrCreateNoAlgorithm();
                    bottomEntry.setNoParams(ksAlias);
                    continue;
                }
                ElytronMessages.log.logIgnoredUnrecognizedKeyStoreEntry(ksAlias);
            }
            catch (IOException | NoSuchAlgorithmException | InvalidParameterSpecException e) {
                ElytronMessages.log.logFailedToReadKeyFromKeyStore(e);
            }
        }
    }

    private KeyStore getKeyStoreInstance(String type) throws CredentialStoreException {
        if (this.providers != null) {
            for (Provider p : this.providers) {
                try {
                    return KeyStore.getInstance(type, p);
                }
                catch (KeyStoreException keyStoreException) {
                }
            }
        }
        try {
            if (ElytronMessages.log.isTraceEnabled()) {
                ElytronMessages.log.tracef("Obtaining KeyStore instance of type %s, providers: %s", (Object)type, (Object)Arrays.toString(Security.getProviders()));
            }
            KeyStore ks = KeyStore.getInstance(type);
            ElytronMessages.log.tracef("Obtained KeyStore instance: %s", (Object)ks);
            return ks;
        }
        catch (KeyStoreException e) {
            throw ElytronMessages.log.cannotInitializeCredentialStore(e);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    private void setupExternalStorage(String keyContainingKeyStoreType, Path keyContainingKeyStoreLocation) throws CredentialStoreException {
        KeyStore keyContainingKeyStore = this.getKeyStoreInstance(keyContainingKeyStoreType);
        this.keyStore = this.getKeyStoreInstance("JCEKS");
        this.externalStorage = new ExternalStorage();
        try {
            char[] storePassword = KeyStoreCredentialStore.getStorePassword(this.protectionParameter);
            if (keyContainingKeyStoreLocation != null) {
                try (InputStream is = Files.newInputStream(keyContainingKeyStoreLocation, new OpenOption[0]);){
                    keyContainingKeyStore.load(is, storePassword);
                }
            }
            EmptyProvider emptyProvider = EmptyProvider.getInstance();
            synchronized (emptyProvider) {
                keyContainingKeyStore.load(null, storePassword);
            }
            this.externalStorage.init(this.cryptographicAlgorithm, this.encryptionKeyAlias, keyContainingKeyStore, storePassword, this.keyStore);
        }
        catch (IOException | GeneralSecurityException e) {
            throw ElytronMessages.log.cannotInitializeCredentialStore(e);
        }
    }

    private static char[] getStorePassword(CredentialStore.ProtectionParameter protectionParameter) throws IOException, CredentialStoreException {
        char[] password;
        if (protectionParameter instanceof CredentialStore.CredentialSourceProtectionParameter) {
            password = ((CredentialStore.CredentialSourceProtectionParameter)protectionParameter).getCredentialSource().applyToCredential(PasswordCredential.class, c -> c.getPassword().castAndApply(ClearPassword.class, ClearPassword::getPassword));
        } else {
            if (protectionParameter != null) {
                throw ElytronMessages.log.invalidProtectionParameter(protectionParameter);
            }
            password = null;
        }
        return password;
    }

    private String calculateNewAlias(String alias, Class<? extends Credential> credentialType, String algorithm, AlgorithmParameterSpec parameterSpec) throws CredentialStoreException {
        StringBuilder b = new StringBuilder(64 + alias.length());
        b.append(alias.toLowerCase(Locale.ROOT));
        b.append('/');
        b.append(credentialType.getSimpleName().toLowerCase(Locale.ROOT));
        b.append('/');
        if (algorithm != null) {
            b.append(algorithm.toLowerCase(Locale.ROOT));
            b.append('/');
            if (parameterSpec != null) {
                try {
                    AlgorithmParameters algorithmParameters = AlgorithmParameters.getInstance(algorithm);
                    algorithmParameters.init(parameterSpec);
                    ByteIterator.ofBytes(algorithmParameters.getEncoded()).base32Encode(Alphabet.Base32Alphabet.LOWERCASE, false).drainTo(b);
                }
                catch (IOException | NoSuchAlgorithmException | InvalidParameterSpecException e) {
                    throw ElytronMessages.log.cannotWriteCredentialToStore(e);
                }
            }
        } else {
            b.append('/');
        }
        return b.toString();
    }

    private static String toLowercase(String str) {
        return str.toLowerCase(Locale.ROOT);
    }

    static {
        HashMap<String, Class<? extends Credential>> map = new HashMap<String, Class<? extends Credential>>();
        for (Class type : Arrays.asList(PasswordCredential.class, X509CertificateChainPublicCredential.class, X509CertificateChainPrivateCredential.class, KeyPairCredential.class, PublicKeyCredential.class, SecretKeyCredential.class, BearerTokenCredential.class)) {
            map.put(type.getSimpleName().toLowerCase(Locale.ROOT), type);
        }
        CREDENTIAL_TYPES = map;
    }

    private final class ExternalStorage {
        private int VERSION = 1;
        private int SECRET_KEY_ENTRY_TYPE = 100;
        private static final String DEFAULT_CRYPTOGRAPHIC_ALGORITHM = "AES/CBC/NoPadding";
        private Cipher encrypt;
        private Cipher decrypt;
        private KeyStore dataKeyStore;
        private KeyStore storageSecretKeyStore;
        private SecretKey storageSecretKey;

        private ExternalStorage() {
        }

        void init(String cryptographicAlgorithm, String keyAlias, KeyStore keyStore, char[] keyPassword, KeyStore dataKeyStore) throws CredentialStoreException {
            if (cryptographicAlgorithm == null) {
                cryptographicAlgorithm = DEFAULT_CRYPTOGRAPHIC_ALGORITHM;
            }
            this.storageSecretKeyStore = keyStore;
            this.dataKeyStore = dataKeyStore;
            try {
                this.fetchStorageSecretKey(keyAlias, keyPassword);
                Provider provider = keyStore.getProvider();
                try {
                    this.encrypt = Cipher.getInstance(cryptographicAlgorithm, provider);
                }
                catch (NoSuchAlgorithmException e) {
                    this.encrypt = Cipher.getInstance(cryptographicAlgorithm);
                }
                try {
                    this.decrypt = Cipher.getInstance(cryptographicAlgorithm, provider);
                }
                catch (NoSuchAlgorithmException e) {
                    this.decrypt = Cipher.getInstance(cryptographicAlgorithm);
                }
            }
            catch (IOException | KeyStoreException | NoSuchAlgorithmException | UnrecoverableEntryException | CertificateException | NoSuchPaddingException e) {
                throw new CredentialStoreException(e);
            }
        }

        private void fetchStorageSecretKey(String keyAlias, char[] keyPassword) throws CertificateException, NoSuchAlgorithmException, IOException, CredentialStoreException, UnrecoverableEntryException, KeyStoreException {
            KeyStore.Entry entry = this.storageSecretKeyStore.getEntry(keyAlias, new KeyStore.PasswordProtection(keyPassword));
            if (entry == null) {
                throw ElytronMessages.log.externalStorageKeyDoesNotExist(keyAlias);
            }
            if (!(entry instanceof KeyStore.SecretKeyEntry)) {
                throw ElytronMessages.log.wrongTypeOfExternalStorageKey(keyAlias);
            }
            this.storageSecretKey = ((KeyStore.SecretKeyEntry)entry).getSecretKey();
        }

        void load(InputStream inputStream) throws IOException, GeneralSecurityException {
            this.dataKeyStore.load(null, null);
            ObjectInputStream ois = new ObjectInputStream(inputStream);
            int fileVersion = ois.readInt();
            if (fileVersion == this.VERSION) {
                while (ois.available() > 0) {
                    int entryType = ois.readInt();
                    if (entryType == this.SECRET_KEY_ENTRY_TYPE) {
                        this.loadSecretKey(ois);
                        continue;
                    }
                    throw ElytronMessages.log.unrecognizedEntryType(Integer.toString(entryType));
                }
            } else {
                throw ElytronMessages.log.unexpectedFileVersion(Integer.toString(fileVersion));
            }
            ois.close();
        }

        private void loadSecretKey(ObjectInputStream ois) throws IOException, GeneralSecurityException {
            byte[] encryptedData = this.readBytes(ois);
            byte[] iv = this.readBytes(ois);
            this.decrypt.init(2, (Key)this.storageSecretKey, new IvParameterSpec(iv));
            Assert.checkMaximumParameter("cipher block size", 256, this.decrypt.getBlockSize());
            byte[] unPadded = this.pkcs7UnPad(this.decrypt.doFinal(encryptedData));
            ObjectInputStream entryOis = new ObjectInputStream(new ByteArrayInputStream(unPadded));
            String ksAlias = entryOis.readUTF();
            byte[] encodedSecretKey = this.readBytes(entryOis);
            KeyStore.SecretKeyEntry entry = new KeyStore.SecretKeyEntry(new SecretKeySpec(encodedSecretKey, KeyStoreCredentialStore.DATA_OID));
            this.dataKeyStore.setEntry(ksAlias, entry, KeyStoreCredentialStore.this.convertParameter(KeyStoreCredentialStore.this.protectionParameter));
        }

        private byte[] readBytes(ObjectInputStream ois) throws IOException {
            byte[] data;
            int actualLen;
            int len = ois.readInt();
            if (len != (actualLen = ois.read(data = new byte[len], 0, len))) {
                throw ElytronMessages.log.readBytesMismatch(actualLen, len);
            }
            return data;
        }

        private int writeBytes(byte[] data, ObjectOutputStream oos) throws IOException {
            int len = data.length;
            oos.writeInt(len);
            oos.write(data, 0, len);
            return len;
        }

        void store(OutputStream outputStream) throws IOException, GeneralSecurityException {
            ObjectOutputStream oos = new ObjectOutputStream(outputStream);
            oos.writeInt(this.VERSION);
            Enumeration<String> ksAliases = this.dataKeyStore.aliases();
            while (ksAliases.hasMoreElements()) {
                String alias = ksAliases.nextElement();
                KeyStore.Entry entry = this.dataKeyStore.getEntry(alias, KeyStoreCredentialStore.this.convertParameter(KeyStoreCredentialStore.this.protectionParameter));
                if (entry instanceof KeyStore.SecretKeyEntry) {
                    this.saveSecretKey(alias, oos, (KeyStore.SecretKeyEntry)entry);
                    continue;
                }
                throw ElytronMessages.log.unrecognizedEntryType(entry != null ? entry.getClass().getCanonicalName() : "null");
            }
            oos.flush();
            oos.close();
        }

        private void saveSecretKey(String ksAlias, ObjectOutputStream oos, KeyStore.SecretKeyEntry entry) throws IOException, GeneralSecurityException {
            ByteArrayOutputStream entryData = new ByteArrayOutputStream(1024);
            ObjectOutputStream entryOos = new ObjectOutputStream(entryData);
            entryOos.writeUTF(ksAlias);
            this.writeBytes(entry.getSecretKey().getEncoded(), entryOos);
            entryOos.flush();
            this.encrypt.init(1, (Key)this.storageSecretKey, (AlgorithmParameterSpec)null);
            int blockSize = this.encrypt.getBlockSize();
            if (blockSize == 0) {
                throw ElytronMessages.log.algorithmNotBlockBased(this.encrypt.getAlgorithm());
            }
            Assert.checkMaximumParameter("cipher block size", 256, blockSize);
            byte[] padded = this.pkcs7Pad(entryData.toByteArray(), blockSize);
            byte[] encrypted = this.encrypt.doFinal(padded);
            byte[] iv = this.encrypt.getIV();
            if (iv == null) {
                throw ElytronMessages.log.algorithmNotIV(this.encrypt.getAlgorithm());
            }
            oos.writeInt(this.SECRET_KEY_ENTRY_TYPE);
            this.writeBytes(encrypted, oos);
            this.writeBytes(iv, oos);
        }

        private byte[] pkcs7Pad(byte[] buffer, int blockSize) {
            int len = buffer.length;
            int toFill = blockSize - len % blockSize;
            byte[] padded = Arrays.copyOf(buffer, toFill + len);
            Arrays.fill(padded, len, padded.length, (byte)toFill);
            return padded;
        }

        private byte[] pkcs7UnPad(byte[] buffer) throws BadPaddingException {
            byte last = buffer[buffer.length - 1];
            int i = buffer.length - 2;
            while (buffer[i] == last) {
                --i;
            }
            if (i + 1 + last != buffer.length) {
                throw new BadPaddingException();
            }
            return Arrays.copyOfRange(buffer, 0, i + 1);
        }
    }

    static final class ParamKey {
        private final AlgorithmParameterSpec parameterSpec;
        private final int hashCode;

        ParamKey(AlgorithmParameterSpec parameterSpec) {
            this.parameterSpec = parameterSpec;
            this.hashCode = KeyUtil.parametersHashCode(parameterSpec);
        }

        public int hashCode() {
            return this.hashCode;
        }

        AlgorithmParameterSpec getParameterSpec() {
            return this.parameterSpec;
        }

        int getHashCode() {
            return this.hashCode;
        }
    }

    static final class BottomEntry {
        private final MidEntry midEntry;
        private final String algorithm;
        private final HashMap<ParamKey, String> map = new HashMap(0);
        private String noParams;

        BottomEntry(MidEntry midEntry, String algorithm) {
            this.midEntry = midEntry;
            this.algorithm = algorithm;
        }

        String getAlgorithm() {
            return this.algorithm;
        }

        HashMap<ParamKey, String> getMap() {
            return this.map;
        }

        String getNoParams() {
            return this.noParams;
        }

        String setNoParams(String noParams) {
            try {
                String string = this.noParams;
                return string;
            }
            finally {
                this.noParams = noParams;
            }
        }

        boolean isEmpty() {
            return this.noParams == null && this.map.isEmpty();
        }

        private String removeNoParams() {
            try {
                String string = this.noParams;
                return string;
            }
            finally {
                this.noParams = null;
            }
        }
    }

    static final class MidEntry {
        private final TopEntry topEntry;
        private final Class<? extends Credential> credentialType;
        private final HashMap<String, BottomEntry> map = new HashMap(0);
        private BottomEntry noAlgorithm;

        MidEntry(TopEntry topEntry, Class<? extends Credential> credentialType) {
            this.topEntry = topEntry;
            this.credentialType = credentialType;
        }

        Class<? extends Credential> getCredentialType() {
            return this.credentialType;
        }

        HashMap<String, BottomEntry> getMap() {
            return this.map;
        }

        BottomEntry getNoAlgorithm() {
            return this.noAlgorithm;
        }

        void setNoAlgorithm(BottomEntry noAlgorithm) {
            this.noAlgorithm = noAlgorithm;
        }

        BottomEntry removeNoAlgorithm() {
            try {
                BottomEntry bottomEntry = this.noAlgorithm;
                return bottomEntry;
            }
            finally {
                this.noAlgorithm = null;
            }
        }

        boolean isEmpty() {
            return this.noAlgorithm == null && this.map.isEmpty();
        }

        private BottomEntry getOrCreateNoAlgorithm() {
            BottomEntry noAlgorithm = this.noAlgorithm;
            return noAlgorithm != null ? noAlgorithm : (this.noAlgorithm = new BottomEntry(this, null));
        }
    }

    static final class TopEntry {
        private final String alias;
        private final HashMap<Class<? extends Credential>, MidEntry> map = new HashMap(0);

        TopEntry(String alias) {
            this.alias = alias;
        }

        String getAlias() {
            return this.alias;
        }

        HashMap<Class<? extends Credential>, MidEntry> getMap() {
            return this.map;
        }
    }

    static interface Hold
    extends AutoCloseable {
        @Override
        public void close();
    }
}

