#!/usr/bin/perl

# Z88DK Z80 Macro Assembler
#
# Copyright (C) Gunther Strube, InterLogic 1993-99
# Copyright (C) Paulo Custodio, 2011-2015
# License: The Artistic License 2.0, http://www.perlfoundation.org/artistic_license_2_0
# Repository: https://github.com/pauloscustodio/z88dk-z80asm
#
# Build opcodes.t test code, using Udo Munk's z80pack assembler as a reference implementation

use Modern::Perl;
use File::Basename;
use File::Slurp;
use Iterator::Array::Jagged;
use Iterator::Simple::Lookahead;
use List::Util qw( max );
use Data::Dump 'dump';

our $KEEP_FILES;
$KEEP_FILES	 = grep {/-keep/} @ARGV; 

my $UDOMUNK_ASM = "dev/z80pack-1.21/z80asm/z80asm.exe";
my $Z80EMU_SRCDIR = '../../libsrc/z80_crt0s/z80_emu';
my @Z80EMU = qw(
		rcmx_cpd
		rcmx_cpdr
		rcmx_cpi
		rcmx_cpir
		rcmx_rld
		rcmx_rrd
);
my $OUTPUT = "t/opcodes.t";
my @OUTPUT = <<"END_HEADER";
#!/usr/bin/perl

# generated by $0, do not edit

use Modern::Perl;
use t::TestZ80asm;

END_HEADER

my $asm1 = <<'END_ASM';
        public ZERO
        defc ZERO    = 0
END_ASM

my $INPUT = read_file(dirname($0).'/'.basename($0, '.pl').'.asm');

#------------------------------------------------------------------------------
# Main
#------------------------------------------------------------------------------
for my $rabbit (0, 1) {
	for my $error (0, 1) {
		my $iter = 	format_iter(
					add_hex_iter(
					filter_error_iter( $error, 
					compute_if_iter( {RABBIT => $rabbit}, 
					tokenize_iter(
					expand_iter( 
					filter_rcs_kw( 
					read_iter($INPUT))))))));
		my @asm; 
		push @asm, $_->{text}."\n" while <$iter>;

		# write test code
		if (@asm) {
			push @OUTPUT, "z80asm(\n";
			push @OUTPUT, "    options => \"-l -b".
						  ($rabbit ? " -DRABBIT --RCMX000 -i\".z80emu()" : "\"").",\n";
			unless ($error) {
				push @OUTPUT, "    asm1 => <<'END_ASM',\n";
				push @OUTPUT, $asm1;
				push @OUTPUT, "END_ASM\n";
			}
			push @OUTPUT, "    asm  => <<'END_ASM',\n";		
			push @OUTPUT, @asm;
			push @OUTPUT, "END_ASM\n);\n";
		}
	}
}
write_file($OUTPUT, @OUTPUT);

#------------------------------------------------------------------------------
# Iterator to read input
#------------------------------------------------------------------------------
sub read_iter {
	my($input) = join("\n", @_);
	my @input = split(/\n/, $input);
	return Iterator::Simple::Lookahead->new( sub {
		my $line = shift(@input);
		defined $line or return;
		$line =~ s/\s*$//;
		return $line."\n";
	} );
}

#------------------------------------------------------------------------------
# Iterator to filter lines with RCS keywords
#------------------------------------------------------------------------------
sub filter_rcs_kw {
	my($in) = @_;
	return Iterator::Simple::Lookahead->new( sub {
		while (1) {
			my $line = $in->next or return;
			next if $line =~ /\$(Header|Id|Log).*?\$/;
			return $line;
		}
	} );
}

#------------------------------------------------------------------------------
# Iterator to expand sequences of {a b c} into lines with a,b,c
#------------------------------------------------------------------------------
sub expand_iter {
	my($in) = @_;
	my %neg_flag = qw(  z  nz  nz z
						c  nc  nc c
						po pe  pe po
						p  m   m  p
					);
	my %prefix =   qw(  ix 0DDh
						iy 0FDh
					);

	return Iterator::Simple::Lookahead->new( sub {
		while (1) {
			my $line = $in->next or return;

			my @args = split(/ ( \{ [^\}]+ \} ) /x, $line);		# separate lists
			if (@args == 1) {
				return $line;					# no lists
			}
			else {
				my @out;
				
				# expand each {a b c} into [a,b,c] and {1} into @item_ref
				my @item_ref = (undef);		# $item_ref[1] = item-1-pos
				my $i = 0;
				for (@args) {
					if (/ \{ (.*) \} /x) {
						my @items = split(' ', $1);
						if (@items == 1 && $items[0] =~ /^\d+/) {		# {1}
							$_ = [$_];
						}
						else {
							$_ = \@items;
							push @item_ref, $i;
						}
					}
					else {
						$_ = [$_];
					}
					$i++;
				}
				
				# iterate through lists
				my $iter = Iterator::Array::Jagged->new(data => \@args);
				while (my @set = $iter->next) {
					# expand {1}, {2}, ...
					for (@set) {
						if (/ \{ (\d+) (.*) \} /x) {
							$_ = $set[ $item_ref[ $1 ] ];
							if ($2) {
								# transformations: -XXX -> remove XXX
								my $transf = $2;
								if ($transf =~ /-(.*)/) {
									my $remove = $1;
									s/\Q$remove\E//g;
								}
								# transformations: ! -> negate flag
								elsif ($transf eq '!') {
									defined $neg_flag{$_} or die "flag '$_' not found";
									$_ = $neg_flag{$_};
								}
								# transformations: @ ix -> DD; iy -> FD
								elsif ($transf eq '@') {
									defined $prefix{$_} or die "prefix '$_' not found";
									$_ = $prefix{$_};
								}
								else {
									die "unknown transformation '$transf'";
								}
							}
						}
					}
					push @out, join("", @set);
				}
				
				# push lines to input stream
				$in->unget( @out );
			}
		}
	} );
}

#------------------------------------------------------------------------------
# Iterator to replace each line by components: 
#	text - original assembly text
#	asm  - z80pack assembly text
#	error - error message, if any
#	warn  - warning message, if any
#------------------------------------------------------------------------------
sub tokenize_iter {
	my($in) = @_;
	return Iterator::Simple::Lookahead->new( sub {
		my $line = $in->next or return;
		my $ret = {};
		for ($line) {
			s/\s*$//;
			if (! /^;/) {
				if (/;;\s*error.*/) {
					$ret->{error} = $&;
					$line = $`;
				}
				s/\s*$//;
				if (/;;\s*warn.*/) {
					$ret->{warn} = $&;
					$line = $`;
				}
				s/\s*$//;
				if (/;;(.*)/) {
					$ret->{asm} = $1." ";
					$line = $`;
					$ret->{asm} =~ s/\s*;;/\n/g;		# multi-line
				}
			}
			
			s/\s*$//;
			$ret->{text} = $line;
			$ret->{asm} ||= $ret->{text};

			$ret->{text} =~ s/\t/ /g;				# remove tabs
			$ret->{asm} =~ s/\t/ /g;				# remove tabs
			
			$ret->{asm} = "" if $ret->{error};
		}
		return $ret;
	} );
}
	
#------------------------------------------------------------------------------
# Iterator to handle IF/ELSE/ENDIF based on \%options
# IF must be on column 1 and in upper case
# add state flag: true/false
#------------------------------------------------------------------------------
sub compute_if_iter {
	my($options, $in) = @_;

	my @states = ();
	my $state = 1;
	my $update_state = sub { $state = 1; for (@states) { $state &&= $_ }; };
	
	return Iterator::Simple::Lookahead->new( sub {
		while (1) {
			my $line = $in->next or return;
			for ($line->{text}) {
				$line->{state} = 1;
				if (/^IF\s+(.*)/) {
					my $expr = $1;
					my $not = $expr =~ s/^\s*!\s*//;
					$expr =~ /^(\w+)\s*(;.*)?$/ or die "IF expression must be identifier";
					
					push @states, $options->{uc($1)} || 0;
					$states[-1] = ! $states[-1] if $not;
					
					$line->{asm} = "";		# no IF in z80pack
					$update_state->();
				}
				elsif (/^ELSE/) {
					@states > 0 or die "ELSE without IF";
					$states[-1] = ! $states[-1];
					
					$line->{asm} = "";		# no ELSE in z80pack
					$update_state->();
				}
				elsif (/^ENDIF/) {
					@states > 0 or die "ENDIF without IF";
					pop @states;
					
					$line->{asm} = "";		# no ENDIF in z80pack
					$update_state->();
				}
				else {
					$line->{state} = $state || 0;
					if (! $state) {
						undef $line->{error};
						undef $line->{warn};
					}
				}
				
				$line->{asm} = "" unless $state;
				
				return $line;
			}
		}		
	} );
}

#------------------------------------------------------------------------------
# Iterator to extract only error lines
#------------------------------------------------------------------------------
sub filter_error_iter {
	my($error, $in) = @_;

	return Iterator::Simple::Lookahead->new( sub {
		while (1) {
			my $line = $in->next or return;
			
			return $line if ((!!$error == !!$line->{error})
							 || $line->{text} =~ /^(IF|ELSE|ENDIF)/);
		}
	} );
}

#------------------------------------------------------------------------------
# Call Udo Munk's assembler to compute hex
#------------------------------------------------------------------------------
sub add_hex_iter {
	my($in) = @_;
	
	# slurp whole iterator, number lines
	my @lines;
	while (my $line = $in->()) {
		$line->{line_nr} = scalar(@lines)+1;
		push @lines, $line;
	}
	
	write_z80pack_asm("test.asm", \@lines);
	call_z80pack_asm("test.asm");
	read_z80pack_lis("test.lis", \@lines);

	unlink('test.asm', 'test.lis', 'test.bin') unless $KEEP_FILES;

	return Iterator::Simple::Lookahead->new(@lines);
}

#------------------------------------------------------------------------------
# write a z80pack asm file
#------------------------------------------------------------------------------
sub write_z80pack_asm {
	my($file, $lines) = @_;
	
	my %used_libs; for (@Z80EMU) { $used_libs{$_} = 0; }

	open(my $fh, ">", $file) or die "open $file: $!\n";
	for (@$lines) {
		next if $_->{asm} eq "";
		next if ! $_->{state};
		print $fh ";;LINE ", $_->{line_nr}, "\n", $_->{asm}, "\n";

		if ($_->{asm} =~ /call\s+(\w+)/i && exists $used_libs{$1}) {
			$used_libs{$1} = 1;
		}
	}

	# Append used libraries
	our $label_n;
	for my $lib (@Z80EMU) {
		if ($used_libs{$lib}) {
			my $lib_asm = read_file($Z80EMU_SRCDIR.'/'.$lib.'.asm');
			$label_n++;
			
			# remove invalid asm, make labels local
			for ($lib_asm) {
				s/^\s*(PUBLIC|EXTERN|SECTION).*//igm;
				while (/^\.(\w+)/im) {
					my $label = $1;
					my $new_label = (uc($label) eq uc($lib)) ? $label : $label.$label_n;
					s/^\.$label\b/$new_label:/igm;
					s/\b$label\b/$new_label/igm;
				}
			}
			print $fh $lib_asm;
		}
	}
}

#------------------------------------------------------------------------------
# use z80pack to assemble the asm file
#------------------------------------------------------------------------------
sub call_z80pack_asm {
	my($file) = @_;
	my $bin_file = basename($file, ".asm").".bin";
	
	my $args = "-fb -l -o$bin_file $file";
	
	-f $UDOMUNK_ASM && -x _ or die "cannot find assembler $UDOMUNK_ASM";
	print "$UDOMUNK_ASM $args\n";
	system "$UDOMUNK_ASM $args" and die "$UDOMUNK_ASM $args failed";
}

#------------------------------------------------------------------------------
# read hex between LINE markers
#------------------------------------------------------------------------------
sub read_z80pack_lis {
	my($file, $lines) = @_;

	my $line_nr = 1;
	
	open(my $in, "<", $file) or die "open $file: $!\n";
	while(<$in>) {
		next if /^\f/;
		next if /^Source file:/;
		next if /^Title:/;
		next if /^LOC/;
		next unless /\S/;
		
		if (/^[0-9a-f]{4} (( [0-9a-f]{2}){1,4})/i) {
			$lines->[$line_nr - 1]{bytes} ||= "";
			$lines->[$line_nr - 1]{bytes} .= uc($1);
		}
		elsif (/;;LINE (\d+)/) {
			$line_nr = 0+$1;
		}
	}
}

#------------------------------------------------------------------------------
# format the assembly language code
#------------------------------------------------------------------------------
sub format_iter {
	my($in) = @_;
	
	my $indent = 8;
	return Iterator::Simple::Lookahead->new( sub {
		while (1) {
			my @lines;
			my $line = $in->next or return;
			
			for ($line->{text}) {
				s/\s+$//;
				return $line unless /\S/; 		# only blanks
				return $line if /^;/;			# only comment

				# extract comment
				my $comment;
				if (/\s*(;.*)/) {
					$comment = $1;
					$_ = $`;
				}
				
				# extract label
				my $label;
				if (/^(\w+:?|\.\w+)/) {
					$label = $1;
					$_ = $';
				}
				
				# remove blanks
				s/^\s+//; s/\s+$//;
				
				# format opcode
				s/(\w+)\s+/ sprintf("%-4s ", $1) /e;
					
				# indent
				my $this_indent = $indent;
				if (/^if(def|ndef)?\b|\{/) {
					$indent += 2; 
				}
				elsif (/^else\b/) {			
					$this_indent -= 2; 
				}
				elsif (/^endif\b|\}/) {	
					$indent -= 2; 
					$this_indent = $indent; 
				}
				else {
				}				
					
				# opcode
				$_ = sprintf("%-*s ", $this_indent-1, $label || "").$_;
					
				# format error/warning - must be on same line as opcode
				if ($line->{error} || $line->{warn}) {
					$_ = sprintf("%-39s %s", $_, $line->{error} || $line->{warn});
				}
				
				# format comments
				if ($comment) {
					if (length($_) > 39) {
						push @lines, $_;
						$_ = sprintf("%-39s %s", "", $comment);
					}
					else {
						$_ = sprintf("%-39s %s", $_, $comment);
					}
				}

				# format bytes
				if ($line->{bytes}) {
					if (length($_) > 39) {
						push @lines, $_;
						$_ = sprintf("%-39s %s", "", ";;".$line->{bytes});
					}
					else {
						$_ = sprintf("%-39s %s", $_, ";;".$line->{bytes});
					}
				}
				
				push @lines, $_;
				$_ = join("\n", @lines);
				return $line;
			}
		}
	} );
}
