/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.monitor;

import java.io.IOException;
import java.util.Iterator;
import java.util.List;
import java.util.Set;
import java.util.concurrent.atomic.AtomicBoolean;

import org.apache.commons.configuration.Configuration;
import org.apache.log4j.Logger;
import org.videolan.vlma.OrderGiver;
import org.videolan.vlma.VLMaService;
import org.videolan.vlma.model.Media;
import org.videolan.vlma.model.Order;
import org.videolan.vlma.model.Program;
import org.videolan.vlma.order.OrderSender;
import org.videolan.vlma.watcher.StreamWatcher;

/**
 * This class is an orders monitoring daemon. It connects to the router and
 * checks that the orders are correctly executed.
 *
 * @author Sylvain Cadilhac <sylv at videolan.org>
 */
public class OrderMonitor implements Monitor {

    private static final Logger logger = Logger.getLogger(OrderMonitor.class);

    private Configuration configuration;

    private VLMaService vlmaService;

    private OrderGiver orderGiver;

    private OrderSender orderSender;

    private Thread orderMonitorThread;

    private Thread orderMonitorDaemonThread;

    private AtomicBoolean shouldRun = new AtomicBoolean(true);

    private StreamWatcher streamWatcher;

    public boolean isMonitoring() {
        return (orderMonitorThread != null) && (orderMonitorThread.isAlive());
    }

    public boolean isDaemonMonitoring() {
        return (orderMonitorDaemonThread != null) && (orderMonitorDaemonThread.isAlive());
    }

    private Runnable orderMonitor = new Runnable() {
        public void run() {
            if (orderGiver.isComputing()) {
                logger.info("Orders are already being given, computation delayed");
                return;
            }
            boolean shouldCompute = false;
            // Assigning programs state
            List<Media> medias = vlmaService.getMedias();
            for (Media media : medias) {
                Program program = media.getProgram();
                if (program != null && program.isTimeToPlay()) {
                    /*
                     * If the program must be played and is not then we have to
                     * recompute.
                     */
                    if (!streamWatcher.isPlayed(program)) {
                        logger.info("Program " + program.getSapName() + " is currently not played.");
                        /*
                         * If the OrderGiver is not computing then it can
                         * modify the data.
                         */
                        synchronized (vlmaService.getOrders()) {
                            Iterator<Order> it = vlmaService.getOrders().iterator();
                            while (it.hasNext()) {
                                Order o = it.next();
                                if(o.getMedias().contains(media)) {
                                    vlmaService.cancelOrder(o);
                                    it.remove();
                                    break;
                                }
                            }
                        }
                        program.setPlayer(null);
                        program.setBroadcastState(false);
                        shouldCompute = true;
                    } else {
                        program.setBroadcastState(true);
                    }
                }
            }
            /*
             * Check if every media which belongs to a order has really a
             * program. If not, then it should compute.
             */
            if (!shouldCompute) {
                for (Order order : vlmaService.getOrders()) {
                    for (Media media : order.getMedias()) {
                        if (media.getProgram() == null) {
                            logger.debug("The media " + media.getName()
                                    + " which belongs to an order has no more a program.");
                            shouldCompute = true;
                            break;
                        }
                    }
                    if (shouldCompute)
                        break;
                }
            }

            /*
             * Cancel orders that have been given but do not seem to be streamed
             */
            Set<Order> ordersToCancel = vlmaService.getOrdersToCancel();
            Iterator<Order> orderIt = ordersToCancel.iterator();
            while (orderIt.hasNext()) {
                Order order = orderIt.next();
                try {
                    orderSender.stop(order);
                    synchronized (ordersToCancel) {
                        orderIt.remove();
                    }
                    logger.debug("Order " + order + " cancelled");
                } catch (IOException e) {
                    // The order could not be stopped, so keep it in the set
                }
            }

            /*
             * Something is wrong ! Ask for a new computation.
             */
            if (shouldCompute) {
                logger.info("Ask for programs reassignment.");
                vlmaService.giveOrders();
            }
        }
    };


    public synchronized void startOrderMonitoringThread() {
        if (!isMonitoring()) {
            orderMonitorThread = new Thread(orderMonitor);
            orderMonitorThread.setName("OrderMonitorThread");
            orderMonitorThread.start();
        }
    }

    private Runnable orderMonitorDaemon = new Runnable() {
        public void run() {
            shouldRun.set(true);
            while (shouldRun.get()) {
                // Wait before looping
                try {
                    Thread.sleep(1000 * configuration.getInt("vlma.monitor.order.interval"));
                } catch (InterruptedException e) {

                }
                if (shouldRun.get())
                    startOrderMonitoringThread();
            }
            logger.debug("OrderMonitor thread stopped.");
        }
    };


    public synchronized void start() {
        if (!isDaemonMonitoring()) {
            logger.info("Starting " + this.getClass().getSimpleName());
            orderMonitorDaemonThread = new Thread(orderMonitorDaemon);
            orderMonitorDaemonThread.setName("OrderMonitorDaemonThread");
            orderMonitorDaemonThread.start();
        }
    }

    public synchronized void stop() {
        logger.info("Stopping " + this.getClass().getSimpleName());
        shouldRun.set(false);
        orderMonitorDaemonThread.interrupt();
    }

    /**
     * Sets the VLMa service.
     *
     * @param vlmaService the vlmaService to set
     */
    public void setVlmaService(VLMaService vlmaService) {
        this.vlmaService = vlmaService;
    }

    /**
     * Sets the OrderGiver implementation to use.
     *
     * @param orderGiver the order giver to set
     */
    public void setOrderGiver(OrderGiver orderGiver) {
        this.orderGiver = orderGiver;
    }

    /**
     * Sets the orderSender.
     *
     * @param orderSender the orderSender to set
     */
    public void setOrderSender(OrderSender orderSender) {
        this.orderSender = orderSender;
    }

    /**
     * Sets the streamWatcher.
     *
     * @param streamWatcher the streamWatcher to set
     */
    public void setStreamWatcher(StreamWatcher streamWatcher) {
        this.streamWatcher = streamWatcher;
    }

    /**
     * Sets the configuration.
     *
     * @param configuration the configuration to set
     */
    public void setConfiguration(Configuration configuration) {
        this.configuration = configuration;
    }

}
