/*
 * Copyright (C) 2006-2008 the VideoLAN team
 *
 * This file is part of VLMa.
 *
 * VLMa is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 2 of the License, or
 * (at your option) any later version.
 *
 * VLMa is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with VLMa. If not, see <http://www.gnu.org/licenses/>.
 *
 */

package org.videolan.vlma.watcher;

import java.io.IOException;
import java.io.InterruptedIOException;
import java.io.Serializable;
import java.net.DatagramPacket;
import java.net.MulticastSocket;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.videolan.vlma.model.Program;

/**
 * This Watcher direcly connects to the multicast groups to check wether some
 * data is broadcasted. If yes, it considers that the media is broadcasted.
 * This class should never be used directly. Use {@link StreamWatcherDispatcher}
 * instead.
 *
 * @author Adrien Maglo <magsoft at videolan.org>
 */
public class DirectMulticastStreamWatcher implements StreamWatcher, Serializable {

    private static final long serialVersionUID = 6311983374515379831L;

    private static final Logger logger = Logger.getLogger(DirectMulticastStreamWatcher.class);

    /**
     * The thread used to wait.
     */
    private transient Thread waitThread;

    /**
     * The multicast socket timeout
     */
    private static int SOCKET_RECEIVE_TIMEOUT = 500;

    /**
     * The minimal lenght of the string to consider that the media is
     * broadcasted
     */
    private static int BUF_LENGHT_MIN = 300;

    /**
     * The time leaved to the multicast socket to get data.
     */
    private static int WAIT_SPAN = 500;

    /**
     * Is true when the socket try to receive data
     */
    private boolean isReceivingData = false;

    public boolean isPlayed(Program program) {
        if (program.getIp() == null) {
            return false;
        }
        boolean isPlayed = false;
        logger.debug("Joining the multicast group " + program.getIp());
        MulticastSocket s;
        try {
            s = new MulticastSocket(1234);
            s.setSoTimeout(SOCKET_RECEIVE_TIMEOUT);
            s.joinGroup(program.getIp());
            byte[] buf = new byte[1024];
            DatagramPacket recv = new DatagramPacket(buf, buf.length);
            /* Try to receive some data */
            try {
                isReceivingData = true;
                startWaitingForTheSocket();
                long bufLenght = 0;
                while (isReceivingData) {
                    s.receive(recv);
                    bufLenght += recv.getLength();
                    /*
                     * If the minimum data quantity has been received,
                     * then ...
                     */
                    if (bufLenght >= BUF_LENGHT_MIN) {
                        /* It is no more necessary to wait */
                        isReceivingData = false;
                        waitThread.interrupt();
                    }
                }

                /*
                 * This is a dirty test but I haven't seen something
                 * else at this moment
                 */
                if (bufLenght >= BUF_LENGHT_MIN) {
                    /*
                     * If some data has been retrieved, then
                     * the stream must be broadcasted
                     */
                    logger.log(Level.DEBUG, "Some data has been received : " + bufLenght + " bytes.");
                    isPlayed = true;
                } else {
                    logger.log(Level.DEBUG, "Not enough data received (" + bufLenght + " bytes) verifying the program " + program.getSapName());
                }
            } catch (InterruptedIOException e) {
                logger.log(Level.DEBUG, "Socket TimeOut verifying the channel " + program.getSapName() + ".");
            }
            /* Leave the group */
            s.leaveGroup(program.getIp());
        } catch (IOException e) {
            logger.error(e);
        }
        return isPlayed;
    }

    /**
     * This wait WAIT_SPAN miliseconds for the multicast socket to receive some
     * data
     */
    private transient Runnable waiter = new Runnable() {
        public void run() {
            /* We start waiting */
            try {
                Thread.sleep(WAIT_SPAN);
                /* Now the socket must stop receiving data */
                isReceivingData = false;
            } catch (InterruptedException e) { }
        }
    };

    /**
     * Start the waiting thread
     */
    private synchronized void startWaitingForTheSocket() {
        waitThread = new Thread(waiter);
        waitThread.setName("waitThread");
        waitThread.start();
    }

}
