import EventHandler from '../dom/event-handler.js';
import SelectorEngine from '../dom/selector-engine.js';
import Config from './config.js';

/**
 * --------------------------------------------------------------------------
 * Bootstrap Italia (https://italia.github.io/bootstrap-italia/)
 * Authors: https://github.com/italia/bootstrap-italia/blob/main/AUTHORS
 * Licensed under BSD-3-Clause license (https://github.com/italia/bootstrap-italia/blob/main/LICENSE)
 * This a fork of Bootstrap: Initial license and original file name below
 * Bootstrap (v5.2.3): util/focustrap.js
 * Licensed under MIT (https://github.com/twbs/bootstrap/blob/main/LICENSE)
 * --------------------------------------------------------------------------
 */


/**
 * Constants
 */

const NAME = 'focustrap';
const DATA_KEY = 'bs.focustrap';
const EVENT_KEY = `.${DATA_KEY}`;
const EVENT_FOCUSIN = `focusin${EVENT_KEY}`;
const EVENT_KEYDOWN_TAB = `keydown.tab${EVENT_KEY}`;

const TAB_KEY = 'Tab';
const TAB_NAV_FORWARD = 'forward';
const TAB_NAV_BACKWARD = 'backward';

const Default = {
  autofocus: true,
  trapElement: null, // The element to trap focus inside of
  initialFocus: null, // The inside element (optional) to set the focus after trapped
};

const DefaultType = {
  autofocus: 'boolean',
  trapElement: 'element',
  initialFocus: '(null|element|string|function)',
};

/**
 * Class definition
 */

class FocusTrap extends Config {
  constructor(config) {
    super();
    this._config = this._getConfig(config);
    this._isActive = false;
    this._lastTabNavDirection = null;
    this._affectedElements = [];
  }

  // Getters
  static get Default() {
    return Default
  }

  static get DefaultType() {
    return DefaultType
  }

  static get NAME() {
    return NAME
  }

  // Public
  activate() {
    if (this._isActive) {
      return
    }

    if (this._config.autofocus) {
      this._setInitialFocus();
    }

    EventHandler.off(document, EVENT_KEY); // guard against infinite focus loop
    EventHandler.on(document, EVENT_FOCUSIN, (event) => this._handleFocusin(event));
    EventHandler.on(document, EVENT_KEYDOWN_TAB, (event) => this._handleKeydown(event));

    this._isActive = true;
  }

  deactivate() {
    if (!this._isActive) {
      return
    }
    this._isActive = false;

    EventHandler.off(document, EVENT_KEY);
  }

  // Private
  _setInitialFocus() {
    if (this._config.initialFocus) {
      let target;
      if (typeof this._config.initialFocus === 'function') {
        target = this._config.initialFocus();
      } else {
        target = this._config.initialFocus;
      }
      if (target && typeof target.focus === 'function') {
        target.focus();
      } else {
        this._config.trapElement.focus();
      }
    } else {
      this._config.trapElement.focus();
    }
  }

  _handleFocusin(event) {
    const { trapElement } = this._config;

    if (event.target === document || event.target === trapElement || trapElement.contains(event.target)) {
      return
    }

    const elements = SelectorEngine.focusableChildren(trapElement);
    if (elements.length === 0) {
      trapElement.focus();
    } else if (this._lastTabNavDirection === TAB_NAV_BACKWARD) {
      elements[elements.length - 1].focus();
    } else {
      elements[0].focus();
    }
  }

  _handleKeydown(event) {
    if (event.key !== TAB_KEY) {
      return
    }

    this._lastTabNavDirection = event.shiftKey ? TAB_NAV_BACKWARD : TAB_NAV_FORWARD;
  }
}

export { FocusTrap as default };
//# sourceMappingURL=focustrap.js.map
