/*
 * Decompiled with CFR 0.152.
 */
package org.apache.rocketmq.tieredstore.file;

import com.google.common.annotations.VisibleForTesting;
import java.nio.ByteBuffer;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Comparator;
import java.util.HashSet;
import java.util.LinkedList;
import java.util.List;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.CompletionStage;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.locks.ReentrantReadWriteLock;
import java.util.stream.Collectors;
import javax.annotation.Nullable;
import org.apache.rocketmq.logging.org.slf4j.Logger;
import org.apache.rocketmq.logging.org.slf4j.LoggerFactory;
import org.apache.rocketmq.tieredstore.common.AppendResult;
import org.apache.rocketmq.tieredstore.common.BoundaryType;
import org.apache.rocketmq.tieredstore.common.FileSegmentType;
import org.apache.rocketmq.tieredstore.exception.TieredStoreErrorCode;
import org.apache.rocketmq.tieredstore.exception.TieredStoreException;
import org.apache.rocketmq.tieredstore.metadata.FileSegmentMetadata;
import org.apache.rocketmq.tieredstore.metadata.TieredMetadataStore;
import org.apache.rocketmq.tieredstore.provider.FileSegmentAllocator;
import org.apache.rocketmq.tieredstore.provider.TieredFileSegment;
import org.apache.rocketmq.tieredstore.util.TieredStoreUtil;

public class TieredFlatFile {
    private static final Logger logger = LoggerFactory.getLogger((String)"RocketmqTieredStore");
    private final String filePath;
    private final FileSegmentType fileType;
    private final TieredMetadataStore tieredMetadataStore;
    private volatile long baseOffset = -1L;
    private final FileSegmentAllocator fileSegmentAllocator;
    private final List<TieredFileSegment> fileSegmentList;
    private final List<TieredFileSegment> needCommitFileSegmentList;
    private final ReentrantReadWriteLock fileSegmentLock;

    public TieredFlatFile(FileSegmentAllocator fileSegmentAllocator, FileSegmentType fileType, String filePath) {
        this.fileType = fileType;
        this.filePath = filePath;
        this.fileSegmentList = new LinkedList<TieredFileSegment>();
        this.fileSegmentLock = new ReentrantReadWriteLock();
        this.fileSegmentAllocator = fileSegmentAllocator;
        this.needCommitFileSegmentList = new CopyOnWriteArrayList<TieredFileSegment>();
        this.tieredMetadataStore = TieredStoreUtil.getMetadataStore(fileSegmentAllocator.getStoreConfig());
        this.recoverMetadata();
        if (fileType != FileSegmentType.INDEX) {
            this.checkAndFixFileSize();
        }
    }

    public long getBaseOffset() {
        return this.baseOffset;
    }

    public void setBaseOffset(long baseOffset) {
        if (this.fileSegmentList.size() > 0) {
            throw new IllegalStateException("Can not set base offset after file segment has been created");
        }
        this.baseOffset = baseOffset;
    }

    public long getMinOffset() {
        this.fileSegmentLock.readLock().lock();
        try {
            if (this.fileSegmentList.isEmpty()) {
                long l = this.baseOffset;
                return l;
            }
            long l = this.fileSegmentList.get(0).getBaseOffset();
            return l;
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
    }

    public long getCommitOffset() {
        this.fileSegmentLock.readLock().lock();
        try {
            if (this.fileSegmentList.isEmpty()) {
                long l = this.baseOffset;
                return l;
            }
            long l = this.fileSegmentList.get(this.fileSegmentList.size() - 1).getCommitOffset();
            return l;
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
    }

    public long getMaxOffset() {
        this.fileSegmentLock.readLock().lock();
        try {
            if (this.fileSegmentList.isEmpty()) {
                long l = this.baseOffset;
                return l;
            }
            long l = this.fileSegmentList.get(this.fileSegmentList.size() - 1).getMaxOffset();
            return l;
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
    }

    public long getDispatchCommitOffset() {
        this.fileSegmentLock.readLock().lock();
        try {
            if (this.fileSegmentList.isEmpty()) {
                long l = 0L;
                return l;
            }
            long l = this.fileSegmentList.get(this.fileSegmentList.size() - 1).getDispatchCommitOffset();
            return l;
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
    }

    @VisibleForTesting
    public List<TieredFileSegment> getFileSegmentList() {
        return this.fileSegmentList;
    }

    protected void recoverMetadata() {
        this.fileSegmentList.clear();
        this.needCommitFileSegmentList.clear();
        this.tieredMetadataStore.iterateFileSegment(this.filePath, this.fileType, metadata -> {
            if (metadata.getStatus() == 2) {
                return;
            }
            TieredFileSegment segment = this.newSegment(this.fileType, metadata.getBaseOffset(), false);
            segment.initPosition(metadata.getSize());
            segment.setMinTimestamp(metadata.getBeginTimestamp());
            segment.setMaxTimestamp(metadata.getEndTimestamp());
            if (metadata.getStatus() == 1) {
                segment.setFull(false);
            }
            this.fileSegmentList.add(segment);
        });
        if (!this.fileSegmentList.isEmpty()) {
            this.fileSegmentList.sort(Comparator.comparingLong(TieredFileSegment::getBaseOffset));
            this.baseOffset = this.fileSegmentList.get(0).getBaseOffset();
            this.needCommitFileSegmentList.addAll(this.fileSegmentList.stream().filter(segment -> !segment.isFull()).collect(Collectors.toList()));
        }
    }

    private FileSegmentMetadata getOrCreateFileSegmentMetadata(TieredFileSegment fileSegment) {
        FileSegmentMetadata metadata = this.tieredMetadataStore.getFileSegment(fileSegment.getPath(), fileSegment.getFileType(), fileSegment.getBaseOffset());
        if (metadata != null) {
            return metadata;
        }
        metadata = new FileSegmentMetadata(this.filePath, fileSegment.getBaseOffset(), fileSegment.getFileType().getType());
        if (fileSegment.isClosed()) {
            metadata.setStatus(2);
        }
        metadata.setBeginTimestamp(fileSegment.getMinTimestamp());
        metadata.setEndTimestamp(fileSegment.getMaxTimestamp());
        this.tieredMetadataStore.updateFileSegment(metadata);
        return metadata;
    }

    @VisibleForTesting
    public void updateFileSegment(TieredFileSegment fileSegment) {
        FileSegmentMetadata segmentMetadata = this.getOrCreateFileSegmentMetadata(fileSegment);
        if (segmentMetadata.getStatus() == 0 && fileSegment.isFull() && !fileSegment.needCommit()) {
            segmentMetadata.markSealed();
        }
        if (fileSegment.isClosed()) {
            segmentMetadata.setStatus(2);
        }
        segmentMetadata.setSize(fileSegment.getCommitPosition());
        segmentMetadata.setBeginTimestamp(fileSegment.getMinTimestamp());
        segmentMetadata.setEndTimestamp(fileSegment.getMaxTimestamp());
        this.tieredMetadataStore.updateFileSegment(segmentMetadata);
    }

    private void checkAndFixFileSize() {
        for (int i = 1; i < this.fileSegmentList.size(); ++i) {
            TieredFileSegment pre = this.fileSegmentList.get(i - 1);
            TieredFileSegment cur = this.fileSegmentList.get(i);
            if (pre.getCommitOffset() == cur.getBaseOffset()) continue;
            logger.warn("TieredFlatFile#checkAndFixFileSize: file segment has incorrect size: filePath:{}, file type: {}, base offset: {}", new Object[]{this.filePath, this.fileType, pre.getBaseOffset()});
            try {
                long actualSize = pre.getSize();
                if (pre.getBaseOffset() + actualSize != cur.getBaseOffset()) {
                    logger.error("[Bug]TieredFlatFile#checkAndFixFileSize: file segment has incorrect size and can not fix: filePath:{}, file type: {}, base offset: {}, actual size: {}, next file offset: {}", new Object[]{this.filePath, this.fileType, pre.getBaseOffset(), actualSize, cur.getBaseOffset()});
                    continue;
                }
                pre.initPosition(actualSize);
                this.updateFileSegment(pre);
                continue;
            }
            catch (Exception e) {
                logger.error("TieredFlatFile#checkAndFixFileSize: fix file segment size failed: filePath: {}, file type: {}, base offset: {}", new Object[]{this.filePath, this.fileType, pre.getBaseOffset()});
            }
        }
        if (!this.fileSegmentList.isEmpty()) {
            TieredFileSegment lastFile = this.fileSegmentList.get(this.fileSegmentList.size() - 1);
            long lastFileSize = lastFile.getSize();
            if (lastFile.getCommitPosition() != lastFileSize) {
                logger.warn("TieredFlatFile#checkAndFixFileSize: fix last file {} size: origin: {}, actual: {}", new Object[]{lastFile.getPath(), lastFile.getCommitOffset() - lastFile.getBaseOffset(), lastFileSize});
                lastFile.initPosition(lastFileSize);
            }
        }
    }

    private TieredFileSegment newSegment(FileSegmentType fileType, long baseOffset, boolean createMetadata) {
        TieredFileSegment segment = null;
        try {
            segment = this.fileSegmentAllocator.createSegment(fileType, this.filePath, baseOffset);
            if (fileType != FileSegmentType.INDEX) {
                segment.createFile();
            }
            if (createMetadata) {
                this.updateFileSegment(segment);
            }
        }
        catch (Exception e) {
            logger.error("create file segment failed: filePath:{}, file type: {}, base offset: {}", new Object[]{this.filePath, fileType, baseOffset, e});
        }
        return segment;
    }

    public void rollingNewFile() {
        TieredFileSegment segment = this.getFileToWrite();
        segment.setFull();
        this.getFileToWrite();
    }

    public int getFileSegmentCount() {
        return this.fileSegmentList.size();
    }

    @Nullable
    protected TieredFileSegment getFileByIndex(int index) {
        this.fileSegmentLock.readLock().lock();
        try {
            if (index < this.fileSegmentList.size()) {
                TieredFileSegment tieredFileSegment = this.fileSegmentList.get(index);
                return tieredFileSegment;
            }
            TieredFileSegment tieredFileSegment = null;
            return tieredFileSegment;
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected TieredFileSegment getFileToWrite() {
        if (this.baseOffset == -1L) {
            throw new IllegalStateException("need to set base offset before create file segment");
        }
        this.fileSegmentLock.readLock().lock();
        try {
            TieredFileSegment fileSegment;
            if (!this.fileSegmentList.isEmpty() && !(fileSegment = this.fileSegmentList.get(this.fileSegmentList.size() - 1)).isFull()) {
                TieredFileSegment tieredFileSegment = fileSegment;
                return tieredFileSegment;
            }
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
        this.fileSegmentLock.writeLock().lock();
        try {
            long offset = this.baseOffset;
            if (!this.fileSegmentList.isEmpty()) {
                TieredFileSegment segment = this.fileSegmentList.get(this.fileSegmentList.size() - 1);
                if (!segment.isFull()) {
                    TieredFileSegment tieredFileSegment = segment;
                    return tieredFileSegment;
                }
                if (segment.commit()) {
                    try {
                        this.updateFileSegment(segment);
                    }
                    catch (Exception e) {
                        TieredFileSegment tieredFileSegment = segment;
                        this.fileSegmentLock.writeLock().unlock();
                        return tieredFileSegment;
                    }
                } else {
                    TieredFileSegment tieredFileSegment = segment;
                    return tieredFileSegment;
                }
                offset = segment.getMaxOffset();
            }
            TieredFileSegment fileSegment = this.newSegment(this.fileType, offset, true);
            this.fileSegmentList.add(fileSegment);
            this.needCommitFileSegmentList.add(fileSegment);
            Collections.sort(this.fileSegmentList);
            logger.debug("Create a new file segment: baseOffset: {}, file: {}, file type: {}", new Object[]{this.baseOffset, fileSegment.getPath(), this.fileType});
            TieredFileSegment tieredFileSegment = fileSegment;
            return tieredFileSegment;
        }
        finally {
            this.fileSegmentLock.writeLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    @Nullable
    protected TieredFileSegment getFileByTime(long timestamp, BoundaryType boundaryType) {
        this.fileSegmentLock.readLock().lock();
        try {
            List segmentList = this.fileSegmentList.stream().sorted(boundaryType == BoundaryType.UPPER ? Comparator.comparingLong(TieredFileSegment::getMaxTimestamp) : Comparator.comparingLong(TieredFileSegment::getMinTimestamp)).filter(segment -> boundaryType == BoundaryType.UPPER ? segment.getMaxTimestamp() >= timestamp : segment.getMinTimestamp() <= timestamp).collect(Collectors.toList());
            if (!segmentList.isEmpty()) {
                TieredFileSegment tieredFileSegment = boundaryType == BoundaryType.UPPER ? (TieredFileSegment)segmentList.get(0) : (TieredFileSegment)segmentList.get(segmentList.size() - 1);
                return tieredFileSegment;
            }
            TieredFileSegment tieredFileSegment = this.fileSegmentList.isEmpty() ? null : this.fileSegmentList.get(this.fileSegmentList.size() - 1);
            return tieredFileSegment;
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected List<TieredFileSegment> getFileListByTime(long beginTime, long endTime) {
        this.fileSegmentLock.readLock().lock();
        try {
            List<TieredFileSegment> list = this.fileSegmentList.stream().filter(segment -> Math.max(beginTime, segment.getMinTimestamp()) <= Math.min(endTime, segment.getMaxTimestamp())).collect(Collectors.toList());
            return list;
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    protected int getSegmentIndexByOffset(long offset) {
        this.fileSegmentLock.readLock().lock();
        try {
            if (this.fileSegmentList.size() == 0) {
                int n = -1;
                return n;
            }
            int left = 0;
            int right = this.fileSegmentList.size() - 1;
            int mid = (left + right) / 2;
            long firstSegmentOffset = this.fileSegmentList.get(left).getBaseOffset();
            long lastSegmentOffset = this.fileSegmentList.get(right).getCommitOffset();
            long midSegmentOffset = this.fileSegmentList.get(mid).getBaseOffset();
            if (offset < firstSegmentOffset || offset > lastSegmentOffset) {
                int n = -1;
                return n;
            }
            while (left < right - 1) {
                if (offset == midSegmentOffset) {
                    int n = mid;
                    return n;
                }
                if (offset < midSegmentOffset) {
                    right = mid;
                } else {
                    left = mid;
                }
                mid = (left + right) / 2;
                midSegmentOffset = this.fileSegmentList.get(mid).getBaseOffset();
            }
            int n = offset < this.fileSegmentList.get(right).getBaseOffset() ? mid : right;
            return n;
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
    }

    public AppendResult append(ByteBuffer byteBuf) {
        return this.append(byteBuf, Long.MAX_VALUE, false);
    }

    public AppendResult append(ByteBuffer byteBuf, long timeStamp) {
        return this.append(byteBuf, timeStamp, false);
    }

    public AppendResult append(ByteBuffer byteBuf, long timeStamp, boolean commit) {
        TieredFileSegment fileSegment = this.getFileToWrite();
        AppendResult result = fileSegment.append(byteBuf, timeStamp);
        if (commit && result == AppendResult.BUFFER_FULL && fileSegment.commit()) {
            result = fileSegment.append(byteBuf, timeStamp);
        }
        if (result == AppendResult.FILE_FULL) {
            return this.getFileToWrite().append(byteBuf, timeStamp);
        }
        return result;
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void cleanExpiredFile(long expireTimestamp) {
        HashSet needToDeleteSet = new HashSet();
        try {
            this.tieredMetadataStore.iterateFileSegment(this.filePath, this.fileType, metadata -> {
                if (metadata.getEndTimestamp() < expireTimestamp) {
                    needToDeleteSet.add(metadata.getBaseOffset());
                }
            });
        }
        catch (Exception e) {
            logger.error("clean expired failed: filePath: {}, file type: {}, expire timestamp: {}", new Object[]{this.filePath, this.fileType, expireTimestamp});
        }
        if (needToDeleteSet.isEmpty()) {
            return;
        }
        this.fileSegmentLock.writeLock().lock();
        try {
            for (int i = 0; i < this.fileSegmentList.size(); ++i) {
                try {
                    TieredFileSegment fileSegment = this.fileSegmentList.get(i);
                    if (!needToDeleteSet.contains(fileSegment.getBaseOffset())) break;
                    fileSegment.close();
                    this.fileSegmentList.remove(fileSegment);
                    this.needCommitFileSegmentList.remove(fileSegment);
                    --i;
                    this.updateFileSegment(fileSegment);
                    logger.info("expired file {} is been cleaned", (Object)fileSegment.getPath());
                    continue;
                }
                catch (Exception e) {
                    logger.error("clean expired file failed: filePath: {}, file type: {}, expire timestamp: {}", new Object[]{this.filePath, this.fileType, expireTimestamp, e});
                }
            }
            this.baseOffset = this.fileSegmentList.size() > 0 ? this.fileSegmentList.get(0).getBaseOffset() : (this.fileType == FileSegmentType.CONSUME_QUEUE ? -1L : 0L);
        }
        finally {
            this.fileSegmentLock.writeLock().unlock();
        }
    }

    @VisibleForTesting
    protected List<TieredFileSegment> getNeedCommitFileSegmentList() {
        return this.needCommitFileSegmentList;
    }

    public void destroyExpiredFile() {
        try {
            this.tieredMetadataStore.iterateFileSegment(this.filePath, this.fileType, metadata -> {
                if (metadata.getStatus() == 2) {
                    try {
                        TieredFileSegment fileSegment = this.newSegment(this.fileType, metadata.getBaseOffset(), false);
                        fileSegment.destroyFile();
                        if (!fileSegment.exists()) {
                            this.tieredMetadataStore.deleteFileSegment(this.filePath, this.fileType, metadata.getBaseOffset());
                            logger.info("expired file {} is been destroyed", (Object)fileSegment.getPath());
                        }
                    }
                    catch (Exception e) {
                        logger.error("destroy expired failed: file path: {}, file type: {}", new Object[]{this.filePath, this.fileType, e});
                    }
                }
            });
        }
        catch (Exception e) {
            logger.error("destroy expired file failed: file path: {}, file type: {}", (Object)this.filePath, (Object)this.fileType);
        }
    }

    public void commit(boolean sync) {
        ArrayList<CompletionStage> futureList = new ArrayList<CompletionStage>();
        try {
            for (TieredFileSegment segment : this.needCommitFileSegmentList) {
                if (segment.isClosed()) continue;
                futureList.add(segment.commitAsync().thenAccept(success -> {
                    try {
                        this.updateFileSegment(segment);
                    }
                    catch (Exception e) {
                        logger.error("update file segment metadata failed: file path: {}, file type: {}, base offset: {}", new Object[]{this.filePath, this.fileType, segment.getBaseOffset(), e});
                    }
                    if (segment.isFull() && !segment.needCommit()) {
                        this.needCommitFileSegmentList.remove(segment);
                    }
                }));
            }
        }
        catch (Exception e) {
            logger.error("commit file segment failed: topic: {}, queue: {}, file type: {}", new Object[]{this.filePath, this.fileType, e});
        }
        if (sync) {
            CompletableFuture.allOf(futureList.toArray(new CompletableFuture[0])).join();
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public CompletableFuture<ByteBuffer> readAsync(long offset, int length) {
        TieredFileSegment fileSegment1;
        int index = this.getSegmentIndexByOffset(offset);
        if (index == -1) {
            String errorMsg = String.format("TieredFlatFile#readAsync: offset is illegal, file path: %s, file type: %s, start: %d, length: %d, file num: %d", new Object[]{this.filePath, this.fileType, offset, length, this.fileSegmentList.size()});
            logger.error(errorMsg);
            throw new TieredStoreException(TieredStoreErrorCode.ILLEGAL_OFFSET, errorMsg);
        }
        TieredFileSegment fileSegment2 = null;
        this.fileSegmentLock.readLock().lock();
        try {
            fileSegment1 = this.fileSegmentList.get(index);
            if (offset + (long)length > fileSegment1.getCommitOffset() && this.fileSegmentList.size() > index + 1) {
                fileSegment2 = this.fileSegmentList.get(index + 1);
            }
        }
        finally {
            this.fileSegmentLock.readLock().unlock();
        }
        if (fileSegment2 == null) {
            return fileSegment1.readAsync(offset - fileSegment1.getBaseOffset(), length);
        }
        int segment1Length = (int)(fileSegment1.getCommitOffset() - offset);
        return fileSegment1.readAsync(offset - fileSegment1.getBaseOffset(), segment1Length).thenCombine(fileSegment2.readAsync(0L, length - segment1Length), (buffer1, buffer2) -> {
            ByteBuffer compositeBuffer = ByteBuffer.allocate(buffer1.remaining() + buffer2.remaining());
            compositeBuffer.put((ByteBuffer)buffer1).put((ByteBuffer)buffer2);
            compositeBuffer.flip();
            return compositeBuffer;
        });
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public void destroy() {
        this.fileSegmentLock.writeLock().lock();
        try {
            for (TieredFileSegment fileSegment : this.fileSegmentList) {
                fileSegment.close();
                try {
                    this.updateFileSegment(fileSegment);
                }
                catch (Exception e) {
                    logger.error("TieredFlatFile#destroy: mark file segment: {} is deleted failed", (Object)fileSegment.getPath(), (Object)e);
                }
                fileSegment.destroyFile();
            }
            this.fileSegmentList.clear();
            this.needCommitFileSegmentList.clear();
        }
        finally {
            this.fileSegmentLock.writeLock().unlock();
        }
    }
}

